/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.cassandra.concurrent.ExecutorFactory;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogPosition;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.WrappedRunnable;
import org.apache.cassandra.utils.concurrent.UncheckedInterruptedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogArchiver {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogArchiver.class);
    public static final DateTimeFormatter format = DateTimeFormatter.ofPattern("yyyy:MM:dd HH:mm:ss[.[SSSSSS][SSS]]").withZone(ZoneId.of("GMT"));
    private static final String COMMITLOG_ARCHIVNG_PROPERTIES_FILE_NAME = "commitlog_archiving.properties";
    private static final String DELIMITER = ",";
    private static final Pattern NAME = Pattern.compile("%name");
    private static final Pattern PATH = Pattern.compile("%path");
    private static final Pattern FROM = Pattern.compile("%from");
    private static final Pattern TO = Pattern.compile("%to");
    public final Map<String, Future<?>> archivePending = new ConcurrentHashMap();
    private final ExecutorService executor;
    final String archiveCommand;
    final String restoreCommand;
    final String restoreDirectories;
    TimeUnit precision;
    long restorePointInTimeInMicroseconds;
    final CommitLogPosition snapshotCommitLogPosition;

    public CommitLogArchiver(String archiveCommand, String restoreCommand, String restoreDirectories, long restorePointInTimeInMicroseconds, CommitLogPosition snapshotCommitLogPosition, TimeUnit precision) {
        this.archiveCommand = archiveCommand;
        this.restoreCommand = restoreCommand;
        this.restoreDirectories = restoreDirectories;
        this.restorePointInTimeInMicroseconds = restorePointInTimeInMicroseconds;
        this.snapshotCommitLogPosition = snapshotCommitLogPosition;
        this.precision = precision;
        this.executor = !Strings.isNullOrEmpty((String)archiveCommand) ? ExecutorFactory.Global.executorFactory().withJmxInternal().sequential("CommitLogArchiver") : null;
    }

    public static CommitLogArchiver disabled() {
        return new CommitLogArchiver(null, null, null, Long.MAX_VALUE, CommitLogPosition.NONE, TimeUnit.MICROSECONDS);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static CommitLogArchiver construct() {
        Properties commitlogProperties = new Properties();
        try (InputStream stream = CommitLogArchiver.class.getClassLoader().getResourceAsStream(COMMITLOG_ARCHIVNG_PROPERTIES_FILE_NAME);){
            if (stream == null) {
                logger.trace("No {} found; archiving and point-in-time-restoration will be disabled", (Object)COMMITLOG_ARCHIVNG_PROPERTIES_FILE_NAME);
                CommitLogArchiver commitLogArchiver2 = CommitLogArchiver.disabled();
                return commitLogArchiver2;
            }
            commitlogProperties.load(stream);
            CommitLogArchiver commitLogArchiver = CommitLogArchiver.getArchiverFromProperties(commitlogProperties);
            return commitLogArchiver;
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to load commitlog_archiving.properties", e);
        }
    }

    @VisibleForTesting
    static CommitLogArchiver getArchiverFromProperties(Properties commitlogCommands) {
        CommitLogPosition snapshotCommitLogPosition;
        TimeUnit precision;
        assert (!commitlogCommands.isEmpty());
        String archiveCommand = commitlogCommands.getProperty("archive_command");
        String restoreCommand = commitlogCommands.getProperty("restore_command");
        String restoreDirectories = commitlogCommands.getProperty("restore_directories");
        if (restoreDirectories != null && !restoreDirectories.isEmpty()) {
            for (String dir : restoreDirectories.split(DELIMITER)) {
                File directory = new File(dir);
                if (directory.exists() || directory.tryCreateDirectory()) continue;
                throw new RuntimeException("Unable to create directory: " + dir);
            }
        }
        String precisionPropertyValue = commitlogCommands.getProperty("precision", TimeUnit.MICROSECONDS.name());
        try {
            precision = TimeUnit.valueOf(precisionPropertyValue);
        }
        catch (IllegalArgumentException ex) {
            throw new RuntimeException("Unable to parse precision of value " + precisionPropertyValue, ex);
        }
        if (precision == TimeUnit.NANOSECONDS) {
            throw new RuntimeException("NANOSECONDS level precision is not supported.");
        }
        String targetTime = commitlogCommands.getProperty("restore_point_in_time");
        long restorePointInTime = Long.MAX_VALUE;
        try {
            if (!Strings.isNullOrEmpty((String)targetTime)) {
                restorePointInTime = CommitLogArchiver.getRestorationPointInTimeInMicroseconds(targetTime);
            }
        }
        catch (DateTimeParseException e) {
            throw new RuntimeException("Unable to parse restore target time", e);
        }
        String snapshotPosition = commitlogCommands.getProperty("snapshot_commitlog_position");
        try {
            snapshotCommitLogPosition = Strings.isNullOrEmpty((String)snapshotPosition) ? CommitLogPosition.NONE : CommitLogPosition.serializer.fromString(snapshotPosition);
        }
        catch (NumberFormatException | ParseException e) {
            throw new RuntimeException("Unable to parse snapshot commit log position", e);
        }
        return new CommitLogArchiver(archiveCommand, restoreCommand, restoreDirectories, restorePointInTime, snapshotCommitLogPosition, precision);
    }

    public void maybeArchive(final CommitLogSegment segment) {
        if (Strings.isNullOrEmpty((String)this.archiveCommand)) {
            return;
        }
        this.archivePending.put(segment.getName(), this.executor.submit(new WrappedRunnable(){

            @Override
            protected void runMayThrow() throws IOException {
                segment.waitForFinalSync();
                String command = NAME.matcher(CommitLogArchiver.this.archiveCommand).replaceAll(Matcher.quoteReplacement(segment.getName()));
                command = PATH.matcher(command).replaceAll(Matcher.quoteReplacement(segment.getPath()));
                CommitLogArchiver.this.exec(command);
            }
        }));
    }

    public void maybeArchive(String path, String name) {
        if (Strings.isNullOrEmpty((String)this.archiveCommand)) {
            return;
        }
        this.archivePending.put(name, this.executor.submit(() -> {
            try {
                String command = NAME.matcher(this.archiveCommand).replaceAll(Matcher.quoteReplacement(name));
                command = PATH.matcher(command).replaceAll(Matcher.quoteReplacement(path));
                this.exec(command);
            }
            catch (IOException e) {
                logger.warn("Archiving file {} failed, file may have already been archived.", (Object)name, (Object)e);
            }
        }));
    }

    public boolean maybeWaitForArchiving(String name) {
        Future<?> f = this.archivePending.remove(name);
        if (f == null) {
            return true;
        }
        try {
            f.get();
        }
        catch (InterruptedException e) {
            throw new UncheckedInterruptedException(e);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof RuntimeException && e.getCause().getCause() instanceof IOException) {
                logger.error("Looks like the archiving of file {} failed earlier, Cassandra is going to ignore this segment for now.", (Object)name, (Object)e.getCause().getCause());
                return false;
            }
            throw new RuntimeException(e);
        }
        return true;
    }

    public void maybeRestoreArchive() {
        if (Strings.isNullOrEmpty((String)this.restoreDirectories)) {
            return;
        }
        for (String dir : this.restoreDirectories.split(DELIMITER)) {
            File[] files = new File(dir).tryList();
            if (files == null) {
                throw new RuntimeException("Unable to list directory " + dir);
            }
            for (File fromFile : files) {
                File toFile;
                CommitLogDescriptor fromName;
                CommitLogDescriptor fromHeader = CommitLogDescriptor.fromHeader(fromFile, DatabaseDescriptor.getEncryptionContext());
                CommitLogDescriptor commitLogDescriptor = fromName = CommitLogDescriptor.isValid(fromFile.name()) ? CommitLogDescriptor.fromFileName(fromFile.name()) : null;
                if (fromHeader == null && fromName == null) {
                    throw new IllegalStateException("Cannot safely construct descriptor for segment, either from its name or its header: " + fromFile.path());
                }
                if (fromHeader != null && fromName != null && !fromHeader.equalsIgnoringCompression(fromName)) {
                    throw new IllegalStateException(String.format("Cannot safely construct descriptor for segment, as name and header descriptors do not match (%s vs %s): %s", fromHeader, fromName, fromFile.path()));
                }
                if (fromName != null && fromHeader == null) {
                    throw new IllegalStateException("Cannot safely construct descriptor for segment, as name descriptor implies a version that should contain a header descriptor, but that descriptor could not be read: " + fromFile.path());
                }
                CommitLogDescriptor descriptor = fromHeader != null ? fromHeader : fromName;
                if (descriptor.version > CommitLogDescriptor.current_version) {
                    throw new IllegalStateException("Unsupported commit log version: " + descriptor.version);
                }
                if (descriptor.compression != null) {
                    try {
                        CompressionParams.createCompressor(descriptor.compression);
                    }
                    catch (ConfigurationException e) {
                        throw new IllegalStateException("Unknown compression", e);
                    }
                }
                if ((toFile = new File(DatabaseDescriptor.getCommitLogLocation(), descriptor.fileName())).exists()) {
                    logger.trace("Skipping restore of archive {} as the segment already exists in the restore location {}", (Object)fromFile.path(), (Object)toFile.path());
                    continue;
                }
                String command = FROM.matcher(this.restoreCommand).replaceAll(Matcher.quoteReplacement(fromFile.path()));
                command = TO.matcher(command).replaceAll(Matcher.quoteReplacement(toFile.path()));
                try {
                    this.exec(command);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    private void exec(String command) throws IOException {
        ProcessBuilder pb = new ProcessBuilder(command.split(" "));
        pb.redirectErrorStream(true);
        FBUtilities.exec(pb);
    }

    @VisibleForTesting
    public static long getRestorationPointInTimeInMicroseconds(String restorationPointInTime) {
        assert (!Strings.isNullOrEmpty((String)restorationPointInTime)) : "restore_point_in_time is null or empty!";
        Instant instant = format.parse((CharSequence)restorationPointInTime, Instant::from);
        return instant.getEpochSecond() * 1000000L + (long)(instant.getNano() / 1000);
    }

    public long getRestorePointInTimeInMicroseconds() {
        return this.restorePointInTimeInMicroseconds;
    }

    @VisibleForTesting
    public void setRestorePointInTimeInMicroseconds(long restorePointInTimeInMicroseconds) {
        this.restorePointInTimeInMicroseconds = restorePointInTimeInMicroseconds;
    }

    @VisibleForTesting
    public void setPrecision(TimeUnit timeUnit) {
        this.precision = timeUnit;
    }
}

