/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.reads;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.SinglePartitionReadCommand;
import org.apache.cassandra.db.partitions.PartitionIterator;
import org.apache.cassandra.db.transform.DuplicateRowChecker;
import org.apache.cassandra.exceptions.ReadFailureException;
import org.apache.cassandra.exceptions.ReadTimeoutException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.locator.EndpointsForToken;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.locator.Replica;
import org.apache.cassandra.locator.ReplicaCollection;
import org.apache.cassandra.locator.ReplicaPlan;
import org.apache.cassandra.locator.ReplicaPlans;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.RequestCallback;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.reads.AlwaysSpeculativeRetryPolicy;
import org.apache.cassandra.service.reads.DigestResolver;
import org.apache.cassandra.service.reads.NeverSpeculativeRetryPolicy;
import org.apache.cassandra.service.reads.ReadCallback;
import org.apache.cassandra.service.reads.SpeculativeRetryPolicy;
import org.apache.cassandra.service.reads.repair.ReadRepair;
import org.apache.cassandra.tracing.TraceState;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.transport.Dispatcher;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.MonotonicClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractReadExecutor {
    private static final Logger logger = LoggerFactory.getLogger(AbstractReadExecutor.class);
    protected final ReadCommand command;
    private final ReplicaPlan.SharedForTokenRead replicaPlan;
    protected final ReadRepair<EndpointsForToken, ReplicaPlan.ForTokenRead> readRepair;
    protected final DigestResolver<EndpointsForToken, ReplicaPlan.ForTokenRead> digestResolver;
    protected final ReadCallback<EndpointsForToken, ReplicaPlan.ForTokenRead> handler;
    protected final TraceState traceState;
    protected final ColumnFamilyStore cfs;
    protected final Dispatcher.RequestTime requestTime;
    private final int initialDataRequestCount;
    protected volatile PartitionIterator result = null;

    AbstractReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ReplicaPlan.ForTokenRead replicaPlan, int initialDataRequestCount, Dispatcher.RequestTime requestTime) {
        this.command = command;
        this.replicaPlan = ReplicaPlan.shared(replicaPlan);
        this.initialDataRequestCount = initialDataRequestCount;
        this.readRepair = ReadRepair.create(command, this.replicaPlan, requestTime);
        this.digestResolver = new DigestResolver<EndpointsForToken, ReplicaPlan.ForTokenRead>(command, this.replicaPlan, requestTime);
        this.handler = new ReadCallback<EndpointsForToken, ReplicaPlan.ForTokenRead>(this.digestResolver, command, this.replicaPlan, requestTime);
        this.cfs = cfs;
        this.traceState = Tracing.instance.get();
        this.requestTime = requestTime;
        int digestVersion = MessagingService.current_version;
        for (Replica replica : (EndpointsForToken)replicaPlan.contacts()) {
            digestVersion = Math.min(digestVersion, MessagingService.instance().versions.get(replica.endpoint()));
        }
        command.setDigestVersion(digestVersion);
    }

    public DecoratedKey getKey() {
        Preconditions.checkState((boolean)(this.command instanceof SinglePartitionReadCommand), (Object)"Can only get keys for SinglePartitionReadCommand");
        return ((SinglePartitionReadCommand)this.command).partitionKey();
    }

    public ReadRepair<EndpointsForToken, ReplicaPlan.ForTokenRead> getReadRepair() {
        return this.readRepair;
    }

    protected void makeFullDataRequests(ReplicaCollection<?> replicas) {
        assert (Iterables.all(replicas, Replica::isFull));
        this.makeRequests(this.command, replicas);
    }

    protected void makeTransientDataRequests(Iterable<Replica> replicas) {
        this.makeRequests(this.command.copyAsTransientQuery(replicas), replicas);
    }

    protected void makeDigestRequests(Iterable<Replica> replicas) {
        assert (Iterables.all(replicas, Replica::isFull));
        this.makeRequests(this.command.copyAsDigestQuery(replicas), replicas);
    }

    private void makeRequests(ReadCommand readCommand, Iterable<Replica> replicas) {
        boolean hasLocalEndpoint = false;
        Message<ReadCommand> message = null;
        for (Replica replica : replicas) {
            assert (replica.isFull() || readCommand.acceptsTransient());
            InetAddressAndPort endpoint = replica.endpoint();
            if (replica.isSelf()) {
                hasLocalEndpoint = true;
                continue;
            }
            if (this.traceState != null) {
                this.traceState.trace("reading {} from {}", (Object)(readCommand.isDigestQuery() ? "digest" : "data"), (Object)endpoint);
            }
            if (null == message) {
                message = readCommand.createMessage(false, this.requestTime);
            }
            MessagingService.instance().sendWithCallback(message, endpoint, this.handler);
        }
        if (hasLocalEndpoint) {
            logger.trace("reading {} locally", (Object)(readCommand.isDigestQuery() ? "digest" : "data"));
            Stage.READ.maybeExecuteImmediately(new StorageProxy.LocalReadRunnable(readCommand, this.handler, this.requestTime));
        }
    }

    public abstract void maybeTryAdditionalReplicas();

    public void executeAsync() {
        EndpointsForToken selected = (EndpointsForToken)this.replicaPlan().contacts();
        EndpointsForToken fullDataRequests = (EndpointsForToken)selected.filter(Replica::isFull, this.initialDataRequestCount);
        this.makeFullDataRequests(fullDataRequests);
        this.makeTransientDataRequests(selected.filterLazily(Replica::isTransient));
        this.makeDigestRequests(selected.filterLazily(r -> r.isFull() && !fullDataRequests.contains((Replica)r)));
    }

    public static AbstractReadExecutor getReadExecutor(SinglePartitionReadCommand command, ConsistencyLevel consistencyLevel, Dispatcher.RequestTime requestTime) throws UnavailableException {
        Keyspace keyspace = Keyspace.open(command.metadata().keyspace);
        ColumnFamilyStore cfs = keyspace.getColumnFamilyStore(command.metadata().id);
        SpeculativeRetryPolicy retry = cfs.metadata().params.speculativeRetry;
        ReplicaPlan.ForTokenRead replicaPlan = ReplicaPlans.forRead(keyspace, command.partitionKey().getToken(), command.indexQueryPlan(), consistencyLevel, retry);
        if (retry.equals(NeverSpeculativeRetryPolicy.INSTANCE) || consistencyLevel == ConsistencyLevel.EACH_QUORUM) {
            return new NeverSpeculatingReadExecutor(cfs, (ReadCommand)command, replicaPlan, requestTime, false);
        }
        if (((EndpointsForToken)replicaPlan.contacts()).size() == ((EndpointsForToken)replicaPlan.readCandidates()).size()) {
            boolean recordFailedSpeculation = consistencyLevel != ConsistencyLevel.ALL;
            return new NeverSpeculatingReadExecutor(cfs, (ReadCommand)command, replicaPlan, requestTime, recordFailedSpeculation);
        }
        if (retry.equals(AlwaysSpeculativeRetryPolicy.INSTANCE)) {
            return new AlwaysSpeculatingReadExecutor(cfs, command, replicaPlan, requestTime);
        }
        return new SpeculatingReadExecutor(cfs, command, replicaPlan, requestTime);
    }

    public boolean hasLocalRead() {
        return this.replicaPlan().lookup(FBUtilities.getBroadcastAddressAndPort()) != null;
    }

    boolean shouldSpeculateAndMaybeWait() {
        long now = MonotonicClock.Global.preciseTime.now();
        long sampleLatencyNanos = TimeUnit.MICROSECONDS.toNanos(this.cfs.sampleReadLatencyMicros);
        if (sampleLatencyNanos > this.command.getTimeout(TimeUnit.NANOSECONDS)) {
            logger.trace("Decided not to speculate as {}ns > {}ns", (Object)sampleLatencyNanos, (Object)this.command.getTimeout(TimeUnit.NANOSECONDS));
            return false;
        }
        if (now + sampleLatencyNanos > this.requestTime.clientDeadline()) {
            logger.trace("Decided not to speculate as native transport timeout will be reached before speculating");
            return false;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Awaiting {}ns before speculating", (Object)sampleLatencyNanos);
        }
        return !this.handler.awaitUntil(this.requestTime.startedAtNanos() + sampleLatencyNanos);
    }

    ReplicaPlan.ForTokenRead replicaPlan() {
        return this.replicaPlan.get();
    }

    void onReadTimeout() {
    }

    public void setResult(PartitionIterator result) {
        Preconditions.checkState((this.result == null ? 1 : 0) != 0, (Object)"Result can only be set once");
        this.result = DuplicateRowChecker.duringRead(result, ((EndpointsForToken)this.replicaPlan.get().readCandidates()).endpointList());
    }

    public void awaitResponses() throws ReadTimeoutException {
        this.awaitResponses(false);
    }

    public void awaitResponses(boolean logBlockingReadRepairAttempt) throws ReadTimeoutException {
        try {
            this.handler.awaitResults();
            assert (this.digestResolver.isDataPresent()) : "awaitResults returned with no data present.";
        }
        catch (ReadTimeoutException e) {
            try {
                this.onReadTimeout();
            }
            finally {
                throw e;
            }
        }
        if (this.digestResolver.responsesMatch()) {
            this.setResult(this.digestResolver.getData());
        } else {
            Tracing.trace("Digest mismatch: Mismatch for key {}", (Object)this.getKey());
            this.readRepair.startRepair(this.digestResolver, this::setResult);
            if (logBlockingReadRepairAttempt) {
                logger.info("Blocking Read Repair triggered for query [{}] at CL.{} with endpoints {}", new Object[]{this.command.toCQLString(), this.replicaPlan().consistencyLevel(), this.replicaPlan().contacts()});
            }
        }
    }

    public void awaitReadRepair() throws ReadTimeoutException {
        try {
            this.readRepair.awaitReads();
        }
        catch (ReadTimeoutException e) {
            if (Tracing.isTracing()) {
                Tracing.trace("Timed out waiting on digest mismatch repair requests");
            } else {
                logger.trace("Timed out waiting on digest mismatch repair requests");
            }
            throw new ReadTimeoutException(this.replicaPlan().consistencyLevel(), this.handler.blockFor - 1, this.handler.blockFor, true);
        }
    }

    boolean isDone() {
        return this.result != null;
    }

    public void maybeSendAdditionalDataRequests() {
        if (this.isDone()) {
            return;
        }
        this.readRepair.maybeSendAdditionalReads();
    }

    public PartitionIterator getResult() throws ReadFailureException, ReadTimeoutException {
        Preconditions.checkState((this.result != null ? 1 : 0) != 0, (Object)"Result must be set first");
        return this.result;
    }

    private static class AlwaysSpeculatingReadExecutor
    extends AbstractReadExecutor {
        public AlwaysSpeculatingReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ReplicaPlan.ForTokenRead replicaPlan, Dispatcher.RequestTime requestTime) {
            super(cfs, command, replicaPlan, ((EndpointsForToken)replicaPlan.contacts()).size() > 1 ? 2 : 1, requestTime);
        }

        @Override
        public void maybeTryAdditionalReplicas() {
        }

        @Override
        public void executeAsync() {
            super.executeAsync();
            this.cfs.metric.speculativeRetries.inc();
        }

        @Override
        void onReadTimeout() {
            this.cfs.metric.speculativeFailedRetries.inc();
        }
    }

    static class SpeculatingReadExecutor
    extends AbstractReadExecutor {
        private volatile boolean speculated = false;

        public SpeculatingReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ReplicaPlan.ForTokenRead replicaPlan, Dispatcher.RequestTime requestTime) {
            super(cfs, command, replicaPlan, replicaPlan.readQuorum() < ((EndpointsForToken)replicaPlan.contacts()).size() ? 2 : 1, requestTime);
        }

        @Override
        public void maybeTryAdditionalReplicas() {
            if (this.shouldSpeculateAndMaybeWait()) {
                ReadCommand retryCommand;
                Replica extraReplica;
                this.cfs.metric.speculativeRetries.inc();
                this.speculated = true;
                ReplicaPlan.ForTokenRead replicaPlan = this.replicaPlan();
                if (this.handler.resolver.isDataPresent()) {
                    extraReplica = replicaPlan.firstUncontactedCandidate(replica -> true);
                    assert (extraReplica != null);
                    retryCommand = extraReplica.isTransient() ? this.command.copyAsTransientQuery(extraReplica) : this.command.copyAsDigestQuery(extraReplica);
                } else {
                    extraReplica = replicaPlan.firstUncontactedCandidate(Replica::isFull);
                    retryCommand = this.command;
                    if (extraReplica == null) {
                        this.cfs.metric.speculativeInsufficientReplicas.inc();
                        return;
                    }
                }
                this.replicaPlan.addToContacts(extraReplica);
                if (this.traceState != null) {
                    this.traceState.trace("speculating read retry on {}", (Object)extraReplica);
                }
                logger.trace("speculating read retry on {}", (Object)extraReplica);
                MessagingService.instance().sendWithCallback(retryCommand.createMessage(false, this.requestTime), extraReplica.endpoint(), (RequestCallback)this.handler);
            }
        }

        @Override
        void onReadTimeout() {
            assert (this.speculated);
            this.cfs.metric.speculativeFailedRetries.inc();
        }
    }

    public static class NeverSpeculatingReadExecutor
    extends AbstractReadExecutor {
        private final boolean logFailedSpeculation;

        public NeverSpeculatingReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ReplicaPlan.ForTokenRead replicaPlan, Dispatcher.RequestTime requestTime, boolean logFailedSpeculation) {
            super(cfs, command, replicaPlan, 1, requestTime);
            this.logFailedSpeculation = logFailedSpeculation;
        }

        @Override
        public void maybeTryAdditionalReplicas() {
            if (this.shouldSpeculateAndMaybeWait() && this.logFailedSpeculation) {
                this.cfs.metric.speculativeInsufficientReplicas.inc();
            }
        }
    }
}

