/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.iceberg.actions;

import java.util.Map;
import org.apache.iceberg.ContentFile;
import org.apache.iceberg.ContentScanTask;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.io.CloseableIterable;

/**
 * Container class holding the output of a {@link FileRewritePlanner#plan()} call. Contains a list
 * of groups (See: {@link RewriteFileGroup} and {@link RewritePositionDeletesGroup}) which are used
 * by engines to rewrite a particular set of files.
 *
 * @param <I> the Java type of the plan info like {@link RewriteDataFiles.FileGroupInfo} or {@link
 *     RewritePositionDeleteFiles.FileGroupInfo}
 * @param <T> the Java type of the input scan tasks (input)
 * @param <F> the Java type of the content files (input and output)
 * @param <G> the Java type of the rewrite file group like {@link RewriteFileGroup} or {@link
 *     RewritePositionDeletesGroup}
 */
public class FileRewritePlan<
    I,
    T extends ContentScanTask<F>,
    F extends ContentFile<F>,
    G extends RewriteGroupBase<I, T, F>> {
  private final CloseableIterable<G> groups;
  private final int totalGroupCount;
  private final Map<StructLike, Integer> groupsInPartition;

  FileRewritePlan(
      CloseableIterable<G> groups,
      int totalGroupCount,
      Map<StructLike, Integer> groupsInPartition) {
    this.groups = groups;
    this.totalGroupCount = totalGroupCount;
    this.groupsInPartition = groupsInPartition;
  }

  /** The stream of the generated {@link RewriteGroupBase}s. */
  public CloseableIterable<G> groups() {
    return groups;
  }

  /** The number of the generated groups in the given partition. */
  public int groupsInPartition(StructLike partition) {
    return groupsInPartition.get(partition);
  }

  /** The total number of the groups generated by this plan. */
  public int totalGroupCount() {
    return totalGroupCount;
  }
}
