"""Unit tests for time series splitters."""

__author__ = ["fkiraly"]

import types

import numpy as np
import pandas as pd
import pytest

from sktime.tests.test_all_estimators import BaseFixtureGenerator, QuickTester
from sktime.utils._testing.hierarchical import _make_hierarchical
from sktime.utils._testing.series import _make_series


class SplitterFixtureGenerator(BaseFixtureGenerator):
    """Fixture generator for splitter tests.

    Fixtures parameterized
    ----------------------
    estimator_class: estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
    estimator_instance: instance of estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
        instances are generated by create_test_instance class method
    """

    # note: this should be separate from TestAllSplitters
    #   additional fixtures, parameters, etc should be added here
    #   TestAllSplitters should contain the tests only

    estimator_type_filter = "splitter"


class TestAllSplitters(SplitterFixtureGenerator, QuickTester):
    """Module level tests for all sktime time series splitters."""

    @pytest.mark.parametrize("hierarchical", [False, True])
    def test_split(self, estimator_instance, hierarchical):
        """Test contract for split."""
        splitter = estimator_instance

        if hierarchical:
            y = _make_hierarchical()
        else:
            y = _make_series()
            # instance splitters do not support single series
            # and should raise informative error message instead
            if splitter.get_tag("split_type") == "instance":
                with pytest.raises(ValueError, match="requires Panel"):
                    list(splitter.split(y))
                return None

        n_splits = splitter.get_n_splits(y)

        assert isinstance(n_splits, int)

        gen_split = splitter.split(y)

        assert isinstance(gen_split, types.GeneratorType)

        splits = list(gen_split)

        assert isinstance(splits, list)

        for x in splits:
            assert isinstance(x, tuple)
            assert len(x) == 2
            assert isinstance(x[0], np.ndarray)
            assert isinstance(x[1], np.ndarray)

            train = x[0]
            test = x[1]

            assert train.ndim == 1
            assert test.ndim == 1
            assert train.dtype in [np.int64, np.int32]
            assert test.dtype in [np.int64, np.int32]

            assert np.all(train < len(y))
            assert np.all(test < len(y))

    @pytest.mark.parametrize("hierarchical", [False, True])
    def test_split_loc(self, estimator_instance, hierarchical):
        """Test contract for split_loc."""
        splitter = estimator_instance

        if hierarchical:
            y = _make_hierarchical()
        else:
            y = _make_series()
            # instance splitters do not support single series
            # and should raise informative error message instead
            if splitter.get_tag("split_type") == "instance":
                with pytest.raises(ValueError, match="requires Panel"):
                    list(splitter.split(y))
                return None

        n_splits = splitter.get_n_splits(y)

        assert isinstance(n_splits, int)

        gen_split = splitter.split_loc(y)

        assert isinstance(gen_split, types.GeneratorType)

        splits = list(gen_split)

        assert isinstance(splits, list)
        assert len(splits) == n_splits

        for x in splits:
            assert isinstance(x, tuple)
            assert len(x) == 2
            assert isinstance(x[0], pd.Index)
            assert isinstance(x[1], pd.Index)

            train = x[0]
            test = x[1]

            assert np.all(train.isin(y.index))
            assert np.all(test.isin(y.index))

    @pytest.mark.parametrize("hierarchical", [False, True])
    def test_split_series(self, estimator_instance, hierarchical):
        """Test contract for split_series."""
        splitter = estimator_instance

        if hierarchical:
            y = _make_hierarchical()
        else:
            y = _make_series()
            # instance splitters do not support single series
            # and should raise informative error message instead
            if splitter.get_tag("split_type") == "instance":
                with pytest.raises(ValueError, match="requires Panel"):
                    list(splitter.split(y))
                return None

        n_splits = splitter.get_n_splits(y)

        assert isinstance(n_splits, int)

        gen_split = splitter.split_series(y)

        assert isinstance(gen_split, types.GeneratorType)

        splits = list(gen_split)

        assert isinstance(splits, list)
        assert len(splits) == n_splits

        if hierarchical:
            expected_type = pd.DataFrame
        else:
            expected_type = pd.Series

        for x in splits:
            assert isinstance(x, tuple)
            assert len(x) == 2
            assert isinstance(x[0], expected_type)
            assert isinstance(x[1], expected_type)
