/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "Adaptor.hh"
#include "ByteRLE.hh"
#include "Compression.hh"
#include "MemoryInputStream.hh"
#include "MemoryOutputStream.hh"
#include "OrcTest.hh"
#include "wrap/gtest-wrapper.h"

#include <iostream>
#include <vector>

namespace orc {

  TEST(ByteRle, simpleTest) {
    const unsigned char buffer[] = {0x61, 0x00, 0xfd, 0x44, 0x45, 0x46};
    std::unique_ptr<ByteRleDecoder> rle =
        createByteRleDecoder(std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
                             getDefaultReaderMetrics());
    std::vector<char> data(103);
    rle->next(data.data(), data.size(), nullptr);

    for (size_t i = 0; i < 100; ++i) {
      EXPECT_EQ(0, data[i]) << "Output wrong at " << i;
    }
    EXPECT_EQ(0x44, data[100]);
    EXPECT_EQ(0x45, data[101]);
    EXPECT_EQ(0x46, data[102]);
  }

  TEST(ByteRle, nullTest) {
    char buffer[258];
    char notNull[266];
    char result[266];
    buffer[0] = static_cast<char>(-128);
    buffer[129] = static_cast<char>(-128);
    for (int i = 0; i < 128; ++i) {
      buffer[1 + i] = static_cast<char>(i);
      buffer[130 + i] = static_cast<char>(128 + i);
    }
    for (int i = 0; i < 266; ++i) {
      notNull[i] = static_cast<char>(i >= 10);
    }
    std::unique_ptr<ByteRleDecoder> rle =
        createByteRleDecoder(std::make_unique<SeekableArrayInputStream>(buffer, sizeof(buffer)),
                             getDefaultReaderMetrics());
    rle->next(result, sizeof(result), notNull);
    for (size_t i = 0; i < sizeof(result); ++i) {
      if (i >= 10) {
        EXPECT_EQ((i - 10) & 0xff, static_cast<int>(result[i]) & 0xff) << "Output wrong at " << i;
      }
    }
  }

  TEST(ByteRle, literalCrossBuffer) {
    const unsigned char buffer[] = {0xf6, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
                                    0x06, 0x07, 0x08, 0x09, 0x07, 0x10};
    std::unique_ptr<ByteRleDecoder> rle = createByteRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer), 6),
        getDefaultReaderMetrics());
    std::vector<char> data(20);
    rle->next(data.data(), data.size(), nullptr);

    for (size_t i = 0; i < 10; ++i) {
      EXPECT_EQ(i, data[i]) << "Output wrong at " << i;
    }
    for (size_t i = 10; i < 20; ++i) {
      EXPECT_EQ(16, data[i]) << "Output wrong at " << i;
    }
  }

  TEST(ByteRle, skipLiteralBufferUnderflowTest) {
    const unsigned char buffer[] = {0xf8, 0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7};
    std::unique_ptr<ByteRleDecoder> rle = createByteRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer), 4),
        getDefaultReaderMetrics());
    std::vector<char> data(8);
    rle->next(data.data(), 3, nullptr);
    EXPECT_EQ(0x0, data[0]);
    EXPECT_EQ(0x1, data[1]);
    EXPECT_EQ(0x2, data[2]);

    rle->skip(2);
    rle->next(data.data(), 3, nullptr);
    EXPECT_EQ(0x5, data[0]);
    EXPECT_EQ(0x6, data[1]);
    EXPECT_EQ(0x7, data[2]);
  }

  TEST(ByteRle, simpleRuns) {
    const unsigned char buffer[] = {0x0d, 0xff, 0x0d, 0xfe, 0x0d, 0xfd};
    std::unique_ptr<ByteRleDecoder> rle =
        createByteRleDecoder(std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
                             getDefaultReaderMetrics());
    std::vector<char> data(16);
    for (size_t i = 0; i < 3; ++i) {
      rle->next(data.data(), data.size(), nullptr);
      for (size_t j = 0; j < data.size(); ++j) {
        EXPECT_EQ(static_cast<char>(-1 - static_cast<int>(i)), data[j])
            << "Output wrong at " << (16 * i + j);
      }
    }
  }

  TEST(ByteRle, splitHeader) {
    const unsigned char buffer[] = {0x00, 0x01, 0xe0, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
                                    0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
                                    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
                                    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20};
    std::unique_ptr<ByteRleDecoder> rle = createByteRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer), 1),
        getDefaultReaderMetrics());
    std::vector<char> data(35);
    rle->next(data.data(), data.size(), nullptr);
    for (size_t i = 0; i < 3; ++i) {
      EXPECT_EQ(1, data[i]) << "Output wrong at " << i;
    }
    for (size_t i = 3; i < data.size(); ++i) {
      EXPECT_EQ(i - 2, data[i]) << "Output wrong at " << i;
    }
  }

  TEST(ByteRle, splitRuns) {
    const unsigned char buffer[] = {0x0d, 0x02, 0xf0, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                                    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10};
    std::unique_ptr<ByteRleDecoder> rle =
        createByteRleDecoder(std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
                             getDefaultReaderMetrics());
    std::vector<char> data(5);
    for (size_t i = 0; i < 3; ++i) {
      rle->next(data.data(), data.size(), nullptr);
      for (size_t j = 0; j < data.size(); ++j) {
        EXPECT_EQ(2, data[j]) << "Output wrong at " << (i * data.size() + j);
      }
    }
    rle->next(data.data(), data.size(), nullptr);
    EXPECT_EQ(2, data[0]) << "Output wrong at 15";
    for (size_t i = 1; i < data.size(); ++i) {
      EXPECT_EQ(i, data[i]) << "Output wrong at " << (15 + i);
    }
    for (size_t i = 0; i < 2; ++i) {
      rle->next(data.data(), data.size(), nullptr);
      for (size_t j = 0; j < data.size(); ++j) {
        EXPECT_EQ(5 * i + j + data.size(), data[j])
            << "Output wrong at " << (20 + data.size() * i + j);
      }
    }
    rle->next(data.data(), 2, nullptr);
    EXPECT_EQ(15, data[0]) << "Output wrong at 30";
    EXPECT_EQ(16, data[1]) << "Output wrong at 31";
  }

  TEST(ByteRle, testNulls) {
    const unsigned char buffer[] = {0xf0, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x3d, 0xdc};
    std::unique_ptr<ByteRleDecoder> rle = createByteRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer), 3),
        getDefaultReaderMetrics());
    std::vector<char> data(16, static_cast<char>(-1));
    std::vector<char> notNull(data.size());
    for (size_t i = 0; i < data.size(); ++i) {
      notNull[i] = (i + 1) % 2;
    }
    for (size_t i = 0; i < 2; ++i) {
      rle->next(data.data(), data.size(), notNull.data());
      for (size_t j = 0; j < data.size(); ++j) {
        if (j % 2 == 0) {
          EXPECT_EQ((i * data.size() + j) / 2, data[j])
              << "Output wrong at " << (i * data.size() + j);
        } else {
          EXPECT_EQ(static_cast<char>(-1), data[j]) << "Output wrong at " << (i * data.size() + j);
        }
      }
    }
    for (size_t i = 0; i < 8; ++i) {
      rle->next(data.data(), data.size(), notNull.data());
      for (size_t j = 0; j < data.size(); ++j) {
        EXPECT_EQ(j % 2 == 0 ? static_cast<char>(-36) : static_cast<char>(-1), data[j])
            << "Output wrong at " << (i * data.size() + j + 32);
      }
    }
  }

  TEST(ByteRle, testAllNulls) {
    const unsigned char buffer[] = {0xf0, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                                    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x3d, 0xdc};
    std::unique_ptr<ByteRleDecoder> rle =
        createByteRleDecoder(std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
                             getDefaultReaderMetrics());
    std::vector<char> data(16, static_cast<char>(-1));
    std::vector<char> allNull(data.size(), 0);
    std::vector<char> noNull(data.size(), 1);
    rle->next(data.data(), data.size(), allNull.data());
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(static_cast<char>(-1), data[i]) << "Output wrong at " << i;
    }
    rle->next(data.data(), data.size(), noNull.data());
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(i, data[i]) << "Output wrong at " << i;
      data[i] = static_cast<char>(-1);
    }
    rle->next(data.data(), data.size(), allNull.data());
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(static_cast<char>(-1), data[i]) << "Output wrong at " << i;
    }
    for (size_t i = 0; i < 4; ++i) {
      rle->next(data.data(), data.size(), noNull.data());
      for (size_t j = 0; j < data.size(); ++j) {
        EXPECT_EQ(static_cast<char>(-36), data[j]) << "Output wrong at " << i;
      }
    }
    rle->next(data.data(), data.size(), allNull.data());
  }

  TEST(ByteRle, testSkip) {
    // the stream generated by Java's TestRunLengthByteReader.testSkips
    // for (int i = 0; i < 2048; ++i) {
    //   if (i < 1024) {
    //     out.write(i / 16);
    //   } else {
    //     out.write(i % 256);
    //   }
    // }
    const unsigned char buffer[] = {
        0xd,  0x0,  0xd,  0x1,  0xd,  0x2,  0xd,  0x3,  0xd,  0x4,  0xd,  0x5,  0xd,  0x6,  0xd,
        0x7,  0xd,  0x8,  0xd,  0x9,  0xd,  0xa,  0xd,  0xb,  0xd,  0xc,  0xd,  0xd,  0xd,  0xe,
        0xd,  0xf,  0xd,  0x10, 0xd,  0x11, 0xd,  0x12, 0xd,  0x13, 0xd,  0x14, 0xd,  0x15, 0xd,
        0x16, 0xd,  0x17, 0xd,  0x18, 0xd,  0x19, 0xd,  0x1a, 0xd,  0x1b, 0xd,  0x1c, 0xd,  0x1d,
        0xd,  0x1e, 0xd,  0x1f, 0xd,  0x20, 0xd,  0x21, 0xd,  0x22, 0xd,  0x23, 0xd,  0x24, 0xd,
        0x25, 0xd,  0x26, 0xd,  0x27, 0xd,  0x28, 0xd,  0x29, 0xd,  0x2a, 0xd,  0x2b, 0xd,  0x2c,
        0xd,  0x2d, 0xd,  0x2e, 0xd,  0x2f, 0xd,  0x30, 0xd,  0x31, 0xd,  0x32, 0xd,  0x33, 0xd,
        0x34, 0xd,  0x35, 0xd,  0x36, 0xd,  0x37, 0xd,  0x38, 0xd,  0x39, 0xd,  0x3a, 0xd,  0x3b,
        0xd,  0x3c, 0xd,  0x3d, 0xd,  0x3e, 0xd,  0x3f, 0x80, 0x0,  0x1,  0x2,  0x3,  0x4,  0x5,
        0x6,  0x7,  0x8,  0x9,  0xa,  0xb,  0xc,  0xd,  0xe,  0xf,  0x10, 0x11, 0x12, 0x13, 0x14,
        0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32,
        0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41,
        0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
        0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e,
        0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d,
        0x7e, 0x7f, 0x80, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
        0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a,
        0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9,
        0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8,
        0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
        0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5,
        0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4,
        0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 0x80, 0x0,  0x1,  0x2,
        0x3,  0x4,  0x5,  0x6,  0x7,  0x8,  0x9,  0xa,  0xb,  0xc,  0xd,  0xe,  0xf,  0x10, 0x11,
        0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
        0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e,
        0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d,
        0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c,
        0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a,
        0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
        0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
        0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6,
        0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5,
        0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4,
        0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3,
        0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2,
        0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1,
        0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 0x80,
        0x0,  0x1,  0x2,  0x3,  0x4,  0x5,  0x6,  0x7,  0x8,  0x9,  0xa,  0xb,  0xc,  0xd,  0xe,
        0xf,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
        0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c,
        0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a,
        0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
        0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
        0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85,
        0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94,
        0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3,
        0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2,
        0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1,
        0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0,
        0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
        0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee,
        0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd,
        0xfe, 0xff, 0x80, 0x0,  0x1,  0x2,  0x3,  0x4,  0x5,  0x6,  0x7,  0x8,  0x9,  0xa,  0xb,
        0xc,  0xd,  0xe,  0xf,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a,
        0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
        0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
        0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56,
        0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65,
        0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,
        0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x80, 0x81, 0x82,
        0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91,
        0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
        0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe,
        0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd,
        0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc,
        0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
        0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa,
        0xfb, 0xfc, 0xfd, 0xfe, 0xff,
    };
    std::unique_ptr<ByteRleDecoder> rle =
        createByteRleDecoder(std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
                             getDefaultReaderMetrics());
    std::vector<char> data(1);
    for (size_t i = 0; i < 2048; i += 10) {
      rle->next(data.data(), data.size(), nullptr);
      EXPECT_EQ(static_cast<char>(i < 1024 ? i / 16 : i & 0xff), data[0])
          << "Output wrong at " << i;
      if (i < 2038) {
        rle->skip(9);
      }
      rle->skip(0);
    }
  }

  TEST(ByteRle, testSeek) {
    // the stream generated by Java's
    // TestRunLengthByteReader.testUncompressedSeek
    // for (int i = 0; i < 2048; ++i) {
    //   if (i < 1024) {
    //     out.write(i / 4);
    //   } else {
    //     out.write(i % 256);
    //   }
    // }
    const unsigned char buffer[] = {
        0x1,  0x0,  0x1,  0x1,  0x1,  0x2,  0x1,  0x3,  0x1,  0x4,  0x1,  0x5,  0x1,  0x6,  0x1,
        0x7,  0x1,  0x8,  0x1,  0x9,  0x1,  0xa,  0x1,  0xb,  0x1,  0xc,  0x1,  0xd,  0x1,  0xe,
        0x1,  0xf,  0x1,  0x10, 0x1,  0x11, 0x1,  0x12, 0x1,  0x13, 0x1,  0x14, 0x1,  0x15, 0x1,
        0x16, 0x1,  0x17, 0x1,  0x18, 0x1,  0x19, 0x1,  0x1a, 0x1,  0x1b, 0x1,  0x1c, 0x1,  0x1d,
        0x1,  0x1e, 0x1,  0x1f, 0x1,  0x20, 0x1,  0x21, 0x1,  0x22, 0x1,  0x23, 0x1,  0x24, 0x1,
        0x25, 0x1,  0x26, 0x1,  0x27, 0x1,  0x28, 0x1,  0x29, 0x1,  0x2a, 0x1,  0x2b, 0x1,  0x2c,
        0x1,  0x2d, 0x1,  0x2e, 0x1,  0x2f, 0x1,  0x30, 0x1,  0x31, 0x1,  0x32, 0x1,  0x33, 0x1,
        0x34, 0x1,  0x35, 0x1,  0x36, 0x1,  0x37, 0x1,  0x38, 0x1,  0x39, 0x1,  0x3a, 0x1,  0x3b,
        0x1,  0x3c, 0x1,  0x3d, 0x1,  0x3e, 0x1,  0x3f, 0x1,  0x40, 0x1,  0x41, 0x1,  0x42, 0x1,
        0x43, 0x1,  0x44, 0x1,  0x45, 0x1,  0x46, 0x1,  0x47, 0x1,  0x48, 0x1,  0x49, 0x1,  0x4a,
        0x1,  0x4b, 0x1,  0x4c, 0x1,  0x4d, 0x1,  0x4e, 0x1,  0x4f, 0x1,  0x50, 0x1,  0x51, 0x1,
        0x52, 0x1,  0x53, 0x1,  0x54, 0x1,  0x55, 0x1,  0x56, 0x1,  0x57, 0x1,  0x58, 0x1,  0x59,
        0x1,  0x5a, 0x1,  0x5b, 0x1,  0x5c, 0x1,  0x5d, 0x1,  0x5e, 0x1,  0x5f, 0x1,  0x60, 0x1,
        0x61, 0x1,  0x62, 0x1,  0x63, 0x1,  0x64, 0x1,  0x65, 0x1,  0x66, 0x1,  0x67, 0x1,  0x68,
        0x1,  0x69, 0x1,  0x6a, 0x1,  0x6b, 0x1,  0x6c, 0x1,  0x6d, 0x1,  0x6e, 0x1,  0x6f, 0x1,
        0x70, 0x1,  0x71, 0x1,  0x72, 0x1,  0x73, 0x1,  0x74, 0x1,  0x75, 0x1,  0x76, 0x1,  0x77,
        0x1,  0x78, 0x1,  0x79, 0x1,  0x7a, 0x1,  0x7b, 0x1,  0x7c, 0x1,  0x7d, 0x1,  0x7e, 0x1,
        0x7f, 0x1,  0x80, 0x1,  0x81, 0x1,  0x82, 0x1,  0x83, 0x1,  0x84, 0x1,  0x85, 0x1,  0x86,
        0x1,  0x87, 0x1,  0x88, 0x1,  0x89, 0x1,  0x8a, 0x1,  0x8b, 0x1,  0x8c, 0x1,  0x8d, 0x1,
        0x8e, 0x1,  0x8f, 0x1,  0x90, 0x1,  0x91, 0x1,  0x92, 0x1,  0x93, 0x1,  0x94, 0x1,  0x95,
        0x1,  0x96, 0x1,  0x97, 0x1,  0x98, 0x1,  0x99, 0x1,  0x9a, 0x1,  0x9b, 0x1,  0x9c, 0x1,
        0x9d, 0x1,  0x9e, 0x1,  0x9f, 0x1,  0xa0, 0x1,  0xa1, 0x1,  0xa2, 0x1,  0xa3, 0x1,  0xa4,
        0x1,  0xa5, 0x1,  0xa6, 0x1,  0xa7, 0x1,  0xa8, 0x1,  0xa9, 0x1,  0xaa, 0x1,  0xab, 0x1,
        0xac, 0x1,  0xad, 0x1,  0xae, 0x1,  0xaf, 0x1,  0xb0, 0x1,  0xb1, 0x1,  0xb2, 0x1,  0xb3,
        0x1,  0xb4, 0x1,  0xb5, 0x1,  0xb6, 0x1,  0xb7, 0x1,  0xb8, 0x1,  0xb9, 0x1,  0xba, 0x1,
        0xbb, 0x1,  0xbc, 0x1,  0xbd, 0x1,  0xbe, 0x1,  0xbf, 0x1,  0xc0, 0x1,  0xc1, 0x1,  0xc2,
        0x1,  0xc3, 0x1,  0xc4, 0x1,  0xc5, 0x1,  0xc6, 0x1,  0xc7, 0x1,  0xc8, 0x1,  0xc9, 0x1,
        0xca, 0x1,  0xcb, 0x1,  0xcc, 0x1,  0xcd, 0x1,  0xce, 0x1,  0xcf, 0x1,  0xd0, 0x1,  0xd1,
        0x1,  0xd2, 0x1,  0xd3, 0x1,  0xd4, 0x1,  0xd5, 0x1,  0xd6, 0x1,  0xd7, 0x1,  0xd8, 0x1,
        0xd9, 0x1,  0xda, 0x1,  0xdb, 0x1,  0xdc, 0x1,  0xdd, 0x1,  0xde, 0x1,  0xdf, 0x1,  0xe0,
        0x1,  0xe1, 0x1,  0xe2, 0x1,  0xe3, 0x1,  0xe4, 0x1,  0xe5, 0x1,  0xe6, 0x1,  0xe7, 0x1,
        0xe8, 0x1,  0xe9, 0x1,  0xea, 0x1,  0xeb, 0x1,  0xec, 0x1,  0xed, 0x1,  0xee, 0x1,  0xef,
        0x1,  0xf0, 0x1,  0xf1, 0x1,  0xf2, 0x1,  0xf3, 0x1,  0xf4, 0x1,  0xf5, 0x1,  0xf6, 0x1,
        0xf7, 0x1,  0xf8, 0x1,  0xf9, 0x1,  0xfa, 0x1,  0xfb, 0x1,  0xfc, 0x1,  0xfd, 0x1,  0xfe,
        0x1,  0xff, 0x80, 0x0,  0x1,  0x2,  0x3,  0x4,  0x5,  0x6,  0x7,  0x8,  0x9,  0xa,  0xb,
        0xc,  0xd,  0xe,  0xf,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a,
        0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
        0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
        0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56,
        0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65,
        0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,
        0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x80, 0x81, 0x82,
        0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91,
        0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
        0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe,
        0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd,
        0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc,
        0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb,
        0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa,
        0xfb, 0xfc, 0xfd, 0xfe, 0xff, 0x80, 0x0,  0x1,  0x2,  0x3,  0x4,  0x5,  0x6,  0x7,  0x8,
        0x9,  0xa,  0xb,  0xc,  0xd,  0xe,  0xf,  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
        0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35,
        0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44,
        0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62,
        0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71,
        0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80,
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e,
        0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d,
        0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac,
        0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb,
        0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca,
        0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9,
        0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8,
        0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
        0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 0x80, 0x0,  0x1,  0x2,  0x3,  0x4,  0x5,
        0x6,  0x7,  0x8,  0x9,  0xa,  0xb,  0xc,  0xd,  0xe,  0xf,  0x10, 0x11, 0x12, 0x13, 0x14,
        0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32,
        0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41,
        0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
        0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e,
        0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d,
        0x7e, 0x7f, 0x80, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b,
        0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a,
        0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9,
        0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8,
        0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
        0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5,
        0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1, 0xf2, 0xf3, 0xf4,
        0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 0x80, 0x0,  0x1,  0x2,
        0x3,  0x4,  0x5,  0x6,  0x7,  0x8,  0x9,  0xa,  0xb,  0xc,  0xd,  0xe,  0xf,  0x10, 0x11,
        0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
        0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e,
        0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d,
        0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c,
        0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a,
        0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
        0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
        0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6,
        0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5,
        0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4,
        0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3,
        0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2,
        0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1,
        0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
    };
    auto stream = std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer));
    const uint64_t fileLocs[] = {
        0,    0,    0,    0,    0,    2,    2,    2,    2,    4,    4,    4,    4,    6,    6,
        6,    6,    8,    8,    8,    8,    10,   10,   10,   10,   12,   12,   12,   12,   14,
        14,   14,   14,   16,   16,   16,   16,   18,   18,   18,   18,   20,   20,   20,   20,
        22,   22,   22,   22,   24,   24,   24,   24,   26,   26,   26,   26,   28,   28,   28,
        28,   30,   30,   30,   30,   32,   32,   32,   32,   34,   34,   34,   34,   36,   36,
        36,   36,   38,   38,   38,   38,   40,   40,   40,   40,   42,   42,   42,   42,   44,
        44,   44,   44,   46,   46,   46,   46,   48,   48,   48,   48,   50,   50,   50,   50,
        52,   52,   52,   52,   54,   54,   54,   54,   56,   56,   56,   56,   58,   58,   58,
        58,   60,   60,   60,   60,   62,   62,   62,   62,   64,   64,   64,   64,   66,   66,
        66,   66,   68,   68,   68,   68,   70,   70,   70,   70,   72,   72,   72,   72,   74,
        74,   74,   74,   76,   76,   76,   76,   78,   78,   78,   78,   80,   80,   80,   80,
        82,   82,   82,   82,   84,   84,   84,   84,   86,   86,   86,   86,   88,   88,   88,
        88,   90,   90,   90,   90,   92,   92,   92,   92,   94,   94,   94,   94,   96,   96,
        96,   96,   98,   98,   98,   98,   100,  100,  100,  100,  102,  102,  102,  102,  104,
        104,  104,  104,  106,  106,  106,  106,  108,  108,  108,  108,  110,  110,  110,  110,
        112,  112,  112,  112,  114,  114,  114,  114,  116,  116,  116,  116,  118,  118,  118,
        118,  120,  120,  120,  120,  122,  122,  122,  122,  124,  124,  124,  124,  126,  126,
        126,  126,  128,  128,  128,  128,  130,  130,  130,  130,  132,  132,  132,  132,  134,
        134,  134,  134,  136,  136,  136,  136,  138,  138,  138,  138,  140,  140,  140,  140,
        142,  142,  142,  142,  144,  144,  144,  144,  146,  146,  146,  146,  148,  148,  148,
        148,  150,  150,  150,  150,  152,  152,  152,  152,  154,  154,  154,  154,  156,  156,
        156,  156,  158,  158,  158,  158,  160,  160,  160,  160,  162,  162,  162,  162,  164,
        164,  164,  164,  166,  166,  166,  166,  168,  168,  168,  168,  170,  170,  170,  170,
        172,  172,  172,  172,  174,  174,  174,  174,  176,  176,  176,  176,  178,  178,  178,
        178,  180,  180,  180,  180,  182,  182,  182,  182,  184,  184,  184,  184,  186,  186,
        186,  186,  188,  188,  188,  188,  190,  190,  190,  190,  192,  192,  192,  192,  194,
        194,  194,  194,  196,  196,  196,  196,  198,  198,  198,  198,  200,  200,  200,  200,
        202,  202,  202,  202,  204,  204,  204,  204,  206,  206,  206,  206,  208,  208,  208,
        208,  210,  210,  210,  210,  212,  212,  212,  212,  214,  214,  214,  214,  216,  216,
        216,  216,  218,  218,  218,  218,  220,  220,  220,  220,  222,  222,  222,  222,  224,
        224,  224,  224,  226,  226,  226,  226,  228,  228,  228,  228,  230,  230,  230,  230,
        232,  232,  232,  232,  234,  234,  234,  234,  236,  236,  236,  236,  238,  238,  238,
        238,  240,  240,  240,  240,  242,  242,  242,  242,  244,  244,  244,  244,  246,  246,
        246,  246,  248,  248,  248,  248,  250,  250,  250,  250,  252,  252,  252,  252,  254,
        254,  254,  254,  256,  256,  256,  256,  258,  258,  258,  258,  260,  260,  260,  260,
        262,  262,  262,  262,  264,  264,  264,  264,  266,  266,  266,  266,  268,  268,  268,
        268,  270,  270,  270,  270,  272,  272,  272,  272,  274,  274,  274,  274,  276,  276,
        276,  276,  278,  278,  278,  278,  280,  280,  280,  280,  282,  282,  282,  282,  284,
        284,  284,  284,  286,  286,  286,  286,  288,  288,  288,  288,  290,  290,  290,  290,
        292,  292,  292,  292,  294,  294,  294,  294,  296,  296,  296,  296,  298,  298,  298,
        298,  300,  300,  300,  300,  302,  302,  302,  302,  304,  304,  304,  304,  306,  306,
        306,  306,  308,  308,  308,  308,  310,  310,  310,  310,  312,  312,  312,  312,  314,
        314,  314,  314,  316,  316,  316,  316,  318,  318,  318,  318,  320,  320,  320,  320,
        322,  322,  322,  322,  324,  324,  324,  324,  326,  326,  326,  326,  328,  328,  328,
        328,  330,  330,  330,  330,  332,  332,  332,  332,  334,  334,  334,  334,  336,  336,
        336,  336,  338,  338,  338,  338,  340,  340,  340,  340,  342,  342,  342,  342,  344,
        344,  344,  344,  346,  346,  346,  346,  348,  348,  348,  348,  350,  350,  350,  350,
        352,  352,  352,  352,  354,  354,  354,  354,  356,  356,  356,  356,  358,  358,  358,
        358,  360,  360,  360,  360,  362,  362,  362,  362,  364,  364,  364,  364,  366,  366,
        366,  366,  368,  368,  368,  368,  370,  370,  370,  370,  372,  372,  372,  372,  374,
        374,  374,  374,  376,  376,  376,  376,  378,  378,  378,  378,  380,  380,  380,  380,
        382,  382,  382,  382,  384,  384,  384,  384,  386,  386,  386,  386,  388,  388,  388,
        388,  390,  390,  390,  390,  392,  392,  392,  392,  394,  394,  394,  394,  396,  396,
        396,  396,  398,  398,  398,  398,  400,  400,  400,  400,  402,  402,  402,  402,  404,
        404,  404,  404,  406,  406,  406,  406,  408,  408,  408,  408,  410,  410,  410,  410,
        412,  412,  412,  412,  414,  414,  414,  414,  416,  416,  416,  416,  418,  418,  418,
        418,  420,  420,  420,  420,  422,  422,  422,  422,  424,  424,  424,  424,  426,  426,
        426,  426,  428,  428,  428,  428,  430,  430,  430,  430,  432,  432,  432,  432,  434,
        434,  434,  434,  436,  436,  436,  436,  438,  438,  438,  438,  440,  440,  440,  440,
        442,  442,  442,  442,  444,  444,  444,  444,  446,  446,  446,  446,  448,  448,  448,
        448,  450,  450,  450,  450,  452,  452,  452,  452,  454,  454,  454,  454,  456,  456,
        456,  456,  458,  458,  458,  458,  460,  460,  460,  460,  462,  462,  462,  462,  464,
        464,  464,  464,  466,  466,  466,  466,  468,  468,  468,  468,  470,  470,  470,  470,
        472,  472,  472,  472,  474,  474,  474,  474,  476,  476,  476,  476,  478,  478,  478,
        478,  480,  480,  480,  480,  482,  482,  482,  482,  484,  484,  484,  484,  486,  486,
        486,  486,  488,  488,  488,  488,  490,  490,  490,  490,  492,  492,  492,  492,  494,
        494,  494,  494,  496,  496,  496,  496,  498,  498,  498,  498,  500,  500,  500,  500,
        502,  502,  502,  502,  504,  504,  504,  504,  506,  506,  506,  506,  508,  508,  508,
        508,  510,  510,  510,  510,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,
        512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  512,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,  641,
        641,  641,  641,  641,  641,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,
        770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  770,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,  899,
        899,  899,  899,  899,  899,  899,  1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028,
        1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1028, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157, 1157,
        1157, 1157, 1157, 1157, 1157, 1157, 1157, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286, 1286,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
        1415, 1415, 1415, 1415, 1415, 1415, 1415, 1415,
    };
    const uint64_t rleLocs[] = {
        0,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,
        4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,   2,   3,   4,   1,
        2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
        20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,
        38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,
        56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,  73,
        74,  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,
        92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102, 103, 104, 105, 106, 107, 108, 109,
        110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127,
        0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,  14,  15,  16,  17,
        18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,
        36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,
        54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,  71,
        72,  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,
        90,  91,  92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102, 103, 104, 105, 106, 107,
        108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125,
        126, 127, 0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,  14,  15,
        16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,
        34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,  50,  51,
        52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
        70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87,
        88,  89,  90,  91,  92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102, 103, 104, 105,
        106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123,
        124, 125, 126, 127, 0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,  12,  13,
        14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,
        32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,  48,  49,
        50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,  66,  67,
        68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,  84,  85,
        86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,  97,  98,  99,  100, 101, 102, 103,
        104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121,
        122, 123, 124, 125, 126, 127, 0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   10,  11,
        12,  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
        30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,  46,  47,
        48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,  64,  65,
        66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,  82,  83,
        84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,  97,  98,  99,  100, 101,
        102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119,
        120, 121, 122, 123, 124, 125, 126, 127, 0,   1,   2,   3,   4,   5,   6,   7,   8,   9,
        10,  11,  12,  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,  26,  27,
        28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,  44,  45,
        46,  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,  63,
        64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,  80,  81,
        82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,  97,  98,  99,
        100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117,
        118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 0,   1,   2,   3,   4,   5,   6,   7,
        8,   9,   10,  11,  12,  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,  24,  25,
        26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,  42,  43,
        44,  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,
        62,  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
        80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,  96,  97,
        98,  99,  100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115,
        116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 0,   1,   2,   3,   4,   5,
        6,   7,   8,   9,   10,  11,  12,  13,  14,  15,  16,  17,  18,  19,  20,  21,  22,  23,
        24,  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,  35,  36,  37,  38,  39,  40,  41,
        42,  43,  44,  45,  46,  47,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,
        60,  61,  62,  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,
        78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,  92,  93,  94,  95,
        96,  97,  98,  99,  100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113,
        114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127,
    };
    std::list<uint64_t> positions[2048];
    for (size_t i = 0; i < 2048; ++i) {
      positions[i].push_back(fileLocs[i]);
      positions[i].push_back(rleLocs[i]);
    }
    std::unique_ptr<ByteRleDecoder> rle =
        createByteRleDecoder(std::move(stream), getDefaultReaderMetrics());
    std::vector<char> data(1);
    for (size_t i = 0; i < 2048; ++i) {
      rle->next(data.data(), 1, nullptr);
      EXPECT_EQ(static_cast<char>(i < 1024 ? i / 4 : i & 0xff), data[0]) << "Output wrong at " << i;
    }
    size_t i = 2048;
    do {
      --i;
      PositionProvider location(positions[i]);
      rle->seek(location);
      rle->next(data.data(), 1, nullptr);
      EXPECT_EQ(static_cast<char>(i < 1024 ? i / 4 : i & 0xff), data[0]) << "Output wrong at " << i;
    } while (i != 0);
  }

  TEST(ByteRle, seekOverEmptyPresentStream) {
    const char* buffer = nullptr;
    std::unique_ptr<ByteRleDecoder> rle = createByteRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, 0, 1), getDefaultReaderMetrics());
    std::list<uint64_t> position(2, 0);
    PositionProvider location(position);
    rle->seek(location);
  }

  TEST(BooleanRle, simpleTest) {
    const unsigned char buffer[] = {0x61, 0xf0, 0xfd, 0x55, 0xAA, 0x55};
    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(50);
    for (size_t i = 0; i < 16; ++i) {
      rle->next(data.data(), data.size(), nullptr);
      for (size_t j = 0; j < data.size(); ++j) {
        const int bitPosn = static_cast<int>(50 * i + j);
        EXPECT_EQ((bitPosn & 0x4) == 0 ? 1 : 0, data[j]) << "Output wrong at " << i << ", " << j;
      }
    }
    rle->next(data.data(), 24, nullptr);
    for (size_t i = 0; i < 3; ++i) {
      for (size_t j = 0; j < 8; ++j) {
        EXPECT_EQ((i % 2) == (j % 2) ? 0 : 1, data[i * 8 + j])
            << "Output wrong at " << i << "," << j;
      }
    }
  }

  TEST(BooleanRle, runsTest) {
    const unsigned char buffer[] = {0xf7, 0xff, 0x80, 0x3f, 0xe0, 0x0f, 0xf8, 0x03, 0xfe, 0x00};
    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(72);
    rle->next(data.data(), data.size(), nullptr);
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(i % 18 < 9 ? 1 : 0, data[i]) << "Output wrong at " << i;
    }
    std::list<uint64_t> position(3, 0);
    PositionProvider location(position);
    rle->seek(location);
    for (size_t i = 0; i < data.size(); ++i) {
      rle->next(data.data(), 1, nullptr);
      EXPECT_EQ(i % 18 < 9 ? 1 : 0, data[0]) << "Output wrong at " << i;
    }
  }

  TEST(BooleanRle, runsTestWithNull) {
    const unsigned char buffer[] = {0xf7, 0xff, 0x80, 0x3f, 0xe0, 0x0f, 0xf8, 0x03, 0xfe, 0x00};
    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(72);
    std::vector<char> notNull(data.size(), 1);
    rle->next(data.data(), data.size(), notNull.data());
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(i % 18 < 9 ? 1 : 0, data[i]) << "Output wrong at " << i;
    }
    std::list<uint64_t> position(3, 0);
    PositionProvider location(position);
    rle->seek(location);
    for (size_t i = 0; i < data.size(); ++i) {
      rle->next(data.data(), 1, notNull.data());
      EXPECT_EQ(i % 18 < 9 ? 1 : 0, data[0]) << "Output wrong at " << i;
    }
  }

  TEST(BooleanRle, skipTest) {
    // stream copied from Java's TestBitFieldReader.testSkips
    // for i in 0..16383
    //   if i < 8192
    //     out.write(i & 1)
    //   else
    //     out.write((i / 3) & 1)
    const unsigned char buffer[] = {
        0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x6f,
        0x55, 0x80, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71};
    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(1);
    for (size_t i = 0; i < 16384; i += 5) {
      rle->next(data.data(), data.size(), nullptr);
      EXPECT_EQ(i < 8192 ? i & 1 : (i / 3) & 1, data[0]) << "Output wrong at " << i;
      if (i < 16379) {
        rle->skip(4);
      }
      rle->skip(0);
    }
  }

  TEST(BooleanRle, skipTestWithNulls) {
    // stream copied from Java's TestBitFieldReader.testSkips
    // for i in 0..16383
    //   if i < 8192
    //     out.write(i & 1)
    //   else
    //     out.write((i / 3) & 1)
    const unsigned char buffer[] = {
        0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x6f,
        0x55, 0x80, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71};
    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(3);
    std::vector<char> someNull(data.size(), 0);
    someNull[1] = 1;
    std::vector<char> allNull(data.size(), 0);
    for (size_t i = 0; i < 16384; i += 5) {
      data.assign(data.size(), static_cast<char>(-1));
      rle->next(data.data(), data.size(), someNull.data());
      EXPECT_EQ(0, data[0]) << "Output wrong at " << i;
      EXPECT_EQ(0, data[2]) << "Output wrong at " << i;
      EXPECT_EQ(i < 8192 ? i & 1 : (i / 3) & 1, data[1]) << "Output wrong at " << i;
      if (i < 16379) {
        rle->skip(4);
      }
      rle->skip(0);
      data.assign(data.size(), static_cast<char>(-1));
      rle->next(data.data(), data.size(), allNull.data());
      for (size_t j = 0; j < data.size(); ++j) {
        EXPECT_EQ(0, data[j]) << "Output wrong at " << i << ", " << j;
      }
    }
  }

  TEST(BooleanRle, seekTest) {
    // stream copied from Java's TestBitFieldReader.testUncompressedSeek
    // for i in 0..16383
    //   if i < 8192
    //     out.write(i & 1)
    //   else
    //     out.write((i / 3) & 1)
    const unsigned char buffer[] = {
        0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x6f,
        0x55, 0x80, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71};
    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(16384);
    rle->next(data.data(), data.size(), nullptr);
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(i < 8192 ? i & 1 : (i / 3) & 1, data[i]) << "Output wrong at " << i;
    }
    // set up all of the positions
    std::list<uint64_t> positions[16384];
    for (uint64_t i = 0; i < 16384; ++i) {
      const uint64_t bytePosn = i / 8;
      // add the stream position
      positions[i].push_back(bytePosn < 1025
                                 ? 2 * (bytePosn / 130)
                                 : (bytePosn < 1152 ? 16 : 145 + 129 * ((bytePosn - 1152) / 128)));
      // add the byte RLE position
      positions[i].push_back(bytePosn < 1025 ? bytePosn % 130 : (bytePosn - 1024) % 128);
      // add the bit position
      positions[i].push_back(i % 8);
    }
    size_t i = 16384;
    do {
      --i;
      PositionProvider location(positions[i]);
      rle->seek(location);
      rle->next(data.data(), 1, nullptr);
      EXPECT_EQ(i < 8192 ? i & 1 : (i / 3) & 1, data[i]) << "Output wrong at " << i;
    } while (i != 0);
  }

  TEST(BooleanRle, seekTestWithNulls) {
    // stream copied from Java's TestBitFieldReader.testUncompressedSeek
    // for i in 0..16383
    //   if i < 8192
    //     out.write(i & 1)
    //   else
    //     out.write((i / 3) & 1)
    const unsigned char buffer[] = {
        0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x7f, 0x55, 0x6f,
        0x55, 0x80, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0x80, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7,
        0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x80, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71,
        0xc7, 0x1c, 0x71, 0xc7, 0x80, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c,
        0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71, 0xc7, 0x1c, 0x71};
    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(16384);
    std::vector<char> allNull(data.size(), 0);
    std::vector<char> noNull(data.size(), 1);
    rle->next(data.data(), data.size(), allNull.data());
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(0, data[i]) << "Output wrong at " << i;
    }
    rle->next(data.data(), data.size(), noNull.data());
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(i < 8192 ? i & 1 : (i / 3) & 1, data[i]) << "Output wrong at " << i;
    }
    // set up all of the positions
    std::list<uint64_t> positions[16384];
    for (uint64_t i = 0; i < 16384; ++i) {
      const uint64_t bytePosn = i / 8;
      // add the stream position
      // add the stream position
      positions[i].push_back(bytePosn < 1025
                                 ? 2 * (bytePosn / 130)
                                 : (bytePosn < 1152 ? 16 : 145 + 129 * ((bytePosn - 1152) / 128)));
      // add the byte RLE position
      positions[i].push_back(bytePosn < 1025 ? bytePosn % 130 : (bytePosn - 1024) % 128);
      // add the bit position
      positions[i].push_back(i % 8);
    }
    size_t i = 16384;
    do {
      --i;
      PositionProvider location(positions[i]);
      rle->seek(location);
      rle->next(data.data(), 1, noNull.data());
      EXPECT_EQ(i < 8192 ? i & 1 : (i / 3) & 1, data[i]) << "Output wrong at " << i;
      data[0] = static_cast<char>(-1);
      rle->next(data.data(), 1, allNull.data());
      EXPECT_EQ(0, data[0]) << "Output wrong at " << i;
    } while (i != 0);
  }

  TEST(BooleanRle, seekBoolAndByteRLE) {
    // ORC-181
    // original data is as follows (1 is true and 0 is false):
    // 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0,
    // 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0,
    // 0, 0, 1, 1
    // The RLE result is 0xf9, 0xf0, 0xf0, 0xf7, 0x1c, 0x71, 0xc1, 0x80
    // The position of the 21st number (index starts from 0) in the RLE result
    // is [0, 2, 5]; the position of the 45th number is [0, 5, 5].
    const char num[] = {1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 1, 1, 0,
                        0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1};
    const unsigned char buffer[] = {0xf9, 0xf0, 0xf0, 0xf7, 0x1c, 0x71, 0xc1, 0x80};

    std::unique_ptr<ByteRleDecoder> rle = createBooleanRleDecoder(
        std::make_unique<SeekableArrayInputStream>(buffer, ARRAY_SIZE(buffer)),
        getDefaultReaderMetrics());
    std::vector<char> data(sizeof(num) / sizeof(char));
    rle->next(data.data(), data.size(), nullptr);
    for (size_t i = 0; i < data.size(); ++i) {
      EXPECT_EQ(num[i], data[i]) << "Output wrong at " << i;
    }

    std::list<uint64_t> pos21st = {0, 2, 5}, pos45th = {0, 5, 5};
    PositionProvider posProvider21st(pos21st), posProvider45th(pos45th);
    char value[1];
    rle->seek(posProvider21st);
    rle->next(value, 1, nullptr);
    EXPECT_EQ(num[21], value[0]);
    rle->seek(posProvider45th);
    rle->next(value, 1, nullptr);
    EXPECT_EQ(num[45], value[0]);
  }

  TEST(BooleanRle, seekAndSkipToEnd) {
    // in total 1024 boolean values which are all true
    constexpr uint64_t numValues = 1024;
    char data[numValues];
    memset(data, 0x01, sizeof(data));

    // create BooleanRleEncoder and encode all 1024 values
    constexpr uint64_t blockSize = 1024, capacity = 1024 * 1024;
    MemoryOutputStream memStream(capacity);
    std::unique_ptr<ByteRleEncoder> encoder = createBooleanRleEncoder(
        createCompressor(CompressionKind_ZSTD, &memStream, CompressionStrategy_COMPRESSION,
                         capacity, blockSize, blockSize, *getDefaultPool(), nullptr));
    encoder->add(data, numValues, nullptr);
    encoder->flush();

    // create BooleanRleDecoder and prepare decoding
    std::unique_ptr<ByteRleDecoder> decoder = createBooleanRleDecoder(
        createDecompressor(
            CompressionKind_ZSTD,
            std::make_unique<SeekableArrayInputStream>(memStream.getData(), memStream.getLength()),
            blockSize, *getDefaultPool(), getDefaultReaderMetrics()),
        getDefaultReaderMetrics());

    // before fix of ORC-470, skip all remaining boolean values will get an
    // exception since BooleanRLEDecoder still tries to read one last byte from
    // underlying input stream even if the requested number of bits are multiple
    // of 8 and then it reads over the end of stream.
    decoder->skip(numValues);

    // as we have reached the end of stream, try to read any data will get exception
    EXPECT_ANY_THROW(decoder->next(data, 1, nullptr));
  }

  TEST(BooleanRle, testSeekWithRemainBitNotZero) {
    MemoryOutputStream memStream(1024 * 1024);

    uint64_t capacity = 500 * 1024;
    uint64_t block = 1024;
    auto outStream = std::make_unique<BufferedOutputStream>(*getDefaultPool(), &memStream, capacity,
                                                            block, nullptr);

    std::unique_ptr<ByteRleEncoder> encoder = createBooleanRleEncoder(std::move(outStream));

    uint64_t numValues = 1779;
    char* data = new char[numValues];
    for (uint64_t i = 0; i < numValues; ++i) {
      data[i] = static_cast<char>(i % 2);
    }
    encoder->add(data, numValues, nullptr);
    encoder->flush();

    auto inStream =
        std::make_unique<SeekableArrayInputStream>(memStream.getData(), memStream.getLength());

    std::unique_ptr<ByteRleDecoder> decoder =
        createBooleanRleDecoder(std::move(inStream), getDefaultReaderMetrics());

    char* decodedData = new char[numValues];
    decoder->next(decodedData, 9, nullptr);

    for (uint64_t i = 0; i < 9; ++i) {
      bool expect = data[i] != 0;
      bool actual = decodedData[i] != 0;
      EXPECT_EQ(expect, actual) << "Output wrong at " << i;
    }

    std::list<uint64_t> positions = {0, 0, 0};
    PositionProvider positionProvider(positions);
    decoder->seek(positionProvider);
    decoder->next(decodedData, numValues, nullptr);
    for (uint64_t i = 0; i < numValues; ++i) {
      bool expect = data[i] != 0;
      bool actual = decodedData[i] != 0;
      EXPECT_EQ(expect, actual) << "Output wrong at " << i;
    }

    delete[] data;
    delete[] decodedData;
  }
}  // namespace orc
