{-# LANGUAGE OverloadedStrings #-}
{- |
   Module      : Text.Pandoc.Readers.Docx.Symbols
   Copyright   : © 2023 John MacFarlane <jgm@berkeley.edu>
   License     : GNU GPL, version 2 or above

   Maintainer  : John MacFarlane <jgm@berkeley.edu>
   Stability   : alpha
   Portability : portable

Symbol tables for MS symbol fonts. This is needed for w:sym.
-}
module Text.Pandoc.Readers.Docx.Symbols ( Font(..),
                                          textToFont,
                                          symbolMap ) where

import Data.Text (Text)
import qualified Data.Map as M

data Font = Symbol | Wingdings | Wingdings2 | Wingdings3 | Webdings
  deriving (Show, Ord, Eq)

textToFont :: Text -> Maybe Font
textToFont "Symbol" = Just Symbol
textToFont "Wingdings" = Just Wingdings
textToFont "Wingdings 2" = Just Wingdings2
textToFont "Wingdings 3" = Just Wingdings3
textToFont "Webdings" = Just Webdings
textToFont _ = Nothing

-- \ Maps font and internal character number to Unicode code point.
symbolMap :: M.Map (Font, Int) Int
symbolMap = M.fromList
  [((Wingdings, 0x21), 0x1F589)
  ,((Wingdings, 0x22), 0x2702)
  ,((Wingdings, 0x23), 0x2701)
  ,((Wingdings, 0x24), 0x1F453)
  ,((Wingdings, 0x25), 0x1F56D)
  ,((Wingdings, 0x26), 0x1F56E)
  ,((Wingdings, 0x27), 0x1F56F)
  ,((Wingdings, 0x28), 0x1F57F)
  ,((Wingdings, 0x29), 0x2706)
  ,((Wingdings, 0x2A), 0x1F582)
  ,((Wingdings, 0x2B), 0x1F583)
  ,((Wingdings, 0x2C), 0x1F4EA)
  ,((Wingdings, 0x2D), 0x1F4EB)
  ,((Wingdings, 0x2E), 0x1F4EC)
  ,((Wingdings, 0x2F), 0x1F4ED)
  ,((Wingdings, 0x30), 0x1F4C1)
  ,((Wingdings, 0x31), 0x1F4C2)
  ,((Wingdings, 0x32), 0x1F4C4)
  ,((Wingdings, 0x33), 0x1F5CF)
  ,((Wingdings, 0x34), 0x1F5D0)
  ,((Wingdings, 0x35), 0x1F5C4)
  ,((Wingdings, 0x36), 0x231B)
  ,((Wingdings, 0x37), 0x1F5AE)
  ,((Wingdings, 0x38), 0x1F5B0)
  ,((Wingdings, 0x39), 0x1F5B2)
  ,((Wingdings, 0x3A), 0x1F5B3)
  ,((Wingdings, 0x3B), 0x1F5B4)
  ,((Wingdings, 0x3C), 0x1F5AB)
  ,((Wingdings, 0x3D), 0x1F5AC)
  ,((Wingdings, 0x3E), 0x2707)
  ,((Wingdings, 0x3F), 0x270D)
  ,((Wingdings, 0x40), 0x1F58E)
  ,((Wingdings, 0x41), 0x270C)
  ,((Wingdings, 0x42), 0x1F44C)
  ,((Wingdings, 0x43), 0x1F44D)
  ,((Wingdings, 0x44), 0x1F44E)
  ,((Wingdings, 0x45), 0x261C)
  ,((Wingdings, 0x46), 0x261E)
  ,((Wingdings, 0x47), 0x261D)
  ,((Wingdings, 0x48), 0x261F)
  ,((Wingdings, 0x49), 0x1F590)
  ,((Wingdings, 0x4A), 0x263A)
  ,((Wingdings, 0x4B), 0x1F610)
  ,((Wingdings, 0x4C), 0x2639)
  ,((Wingdings, 0x4D), 0x1F4A3)
  ,((Wingdings, 0x4E), 0x2620)
  ,((Wingdings, 0x4F), 0x1F3F3)
  ,((Wingdings, 0x50), 0x1F3F1)
  ,((Wingdings, 0x51), 0x2708)
  ,((Wingdings, 0x52), 0x263C)
  ,((Wingdings, 0x53), 0x1F4A7)
  ,((Wingdings, 0x54), 0x2744)
  ,((Wingdings, 0x55), 0x1F546)
  ,((Wingdings, 0x56), 0x271E)
  ,((Wingdings, 0x57), 0x1F548)
  ,((Wingdings, 0x58), 0x2720)
  ,((Wingdings, 0x59), 0x2721)
  ,((Wingdings, 0x5A), 0x262A)
  ,((Wingdings, 0x5B), 0x262F)
  ,((Wingdings, 0x5C), 0x0950)
  ,((Wingdings, 0x5D), 0x2638)
  ,((Wingdings, 0x5E), 0x2648)
  ,((Wingdings, 0x5F), 0x2649)
  ,((Wingdings, 0x60), 0x264A)
  ,((Wingdings, 0x61), 0x264B)
  ,((Wingdings, 0x62), 0x264C)
  ,((Wingdings, 0x63), 0x264D)
  ,((Wingdings, 0x64), 0x264E)
  ,((Wingdings, 0x65), 0x264F)
  ,((Wingdings, 0x66), 0x2650)
  ,((Wingdings, 0x67), 0x2651)
  ,((Wingdings, 0x68), 0x2652)
  ,((Wingdings, 0x69), 0x2653)
  ,((Wingdings, 0x6A), 0x1F670)
  ,((Wingdings, 0x6B), 0x1F675)
  ,((Wingdings, 0x6C), 0x25CF)
  ,((Wingdings, 0x6D), 0x1F53E)
  ,((Wingdings, 0x6E), 0x25A0)
  ,((Wingdings, 0x6F), 0x25A1)
  ,((Wingdings, 0x70), 0x1F790)
  ,((Wingdings, 0x71), 0x2751)
  ,((Wingdings, 0x72), 0x2752)
  ,((Wingdings, 0x73), 0x2B27)
  ,((Wingdings, 0x74), 0x29EB)
  ,((Wingdings, 0x75), 0x25C6)
  ,((Wingdings, 0x76), 0x2756)
  ,((Wingdings, 0x77), 0x2B25)
  ,((Wingdings, 0x78), 0x2327)
  ,((Wingdings, 0x79), 0x2BB9)
  ,((Wingdings, 0x7A), 0x2318)
  ,((Wingdings, 0x7B), 0x1F3F5)
  ,((Wingdings, 0x7C), 0x1F3F6)
  ,((Wingdings, 0x7D), 0x1F676)
  ,((Wingdings, 0x7E), 0x1F677)
  ,((Wingdings, 0x80), 0x24EA)
  ,((Wingdings, 0x81), 0x2460)
  ,((Wingdings, 0x82), 0x2461)
  ,((Wingdings, 0x83), 0x2462)
  ,((Wingdings, 0x84), 0x2463)
  ,((Wingdings, 0x85), 0x2464)
  ,((Wingdings, 0x86), 0x2465)
  ,((Wingdings, 0x87), 0x2466)
  ,((Wingdings, 0x88), 0x2467)
  ,((Wingdings, 0x89), 0x2468)
  ,((Wingdings, 0x8A), 0x2469)
  ,((Wingdings, 0x8B), 0x24FF)
  ,((Wingdings, 0x8C), 0x2776)
  ,((Wingdings, 0x8D), 0x2777)
  ,((Wingdings, 0x8E), 0x2778)
  ,((Wingdings, 0x8F), 0x2779)
  ,((Wingdings, 0x90), 0x277A)
  ,((Wingdings, 0x91), 0x277B)
  ,((Wingdings, 0x92), 0x277C)
  ,((Wingdings, 0x93), 0x277D)
  ,((Wingdings, 0x94), 0x277E)
  ,((Wingdings, 0x95), 0x277F)
  ,((Wingdings, 0x96), 0x1F662)
  ,((Wingdings, 0x97), 0x1F660)
  ,((Wingdings, 0x98), 0x1F661)
  ,((Wingdings, 0x99), 0x1F663)
  ,((Wingdings, 0x9A), 0x1F65E)
  ,((Wingdings, 0x9B), 0x1F65C)
  ,((Wingdings, 0x9C), 0x1F65D)
  ,((Wingdings, 0x9D), 0x1F65F)
  ,((Wingdings, 0x9E), 0x00B7)
  ,((Wingdings, 0x9F), 0x2022)
  ,((Wingdings, 0xA1), 0x26AA)
  ,((Wingdings, 0xA2), 0x1F786)
  ,((Wingdings, 0xA3), 0x1F788)
  ,((Wingdings, 0xA4), 0x25C9)
  ,((Wingdings, 0xA5), 0x25CE)
  ,((Wingdings, 0xA6), 0x1F53F)
  ,((Wingdings, 0xA7), 0x25AA)
  ,((Wingdings, 0xA8), 0x25FB)
  ,((Wingdings, 0xA9), 0x1F7C2)
  ,((Wingdings, 0xAA), 0x2726)
  ,((Wingdings, 0xAB), 0x2605)
  ,((Wingdings, 0xAC), 0x2736)
  ,((Wingdings, 0xAD), 0x2734)
  ,((Wingdings, 0xAE), 0x2739)
  ,((Wingdings, 0xAF), 0x2735)
  ,((Wingdings, 0xB0), 0x2BD0)
  ,((Wingdings, 0xB1), 0x2316)
  ,((Wingdings, 0xB2), 0x27E1)
  ,((Wingdings, 0xB3), 0x2311)
  ,((Wingdings, 0xB4), 0x2BD1)
  ,((Wingdings, 0xB5), 0x272A)
  ,((Wingdings, 0xB6), 0x2730)
  ,((Wingdings, 0xB7), 0x1F550)
  ,((Wingdings, 0xB8), 0x1F551)
  ,((Wingdings, 0xB9), 0x1F552)
  ,((Wingdings, 0xBA), 0x1F553)
  ,((Wingdings, 0xBB), 0x1F554)
  ,((Wingdings, 0xBC), 0x1F555)
  ,((Wingdings, 0xBD), 0x1F556)
  ,((Wingdings, 0xBE), 0x1F557)
  ,((Wingdings, 0xBF), 0x1F558)
  ,((Wingdings, 0xC0), 0x1F559)
  ,((Wingdings, 0xC1), 0x1F55A)
  ,((Wingdings, 0xC2), 0x1F55B)
  ,((Wingdings, 0xC3), 0x2BB0)
  ,((Wingdings, 0xC4), 0x2BB1)
  ,((Wingdings, 0xC5), 0x2BB2)
  ,((Wingdings, 0xC6), 0x2BB3)
  ,((Wingdings, 0xC7), 0x2BB4)
  ,((Wingdings, 0xC8), 0x2BB5)
  ,((Wingdings, 0xC9), 0x2BB6)
  ,((Wingdings, 0xCA), 0x2BB7)
  ,((Wingdings, 0xCB), 0x1F66A)
  ,((Wingdings, 0xCC), 0x1F66B)
  ,((Wingdings, 0xCD), 0x1F655)
  ,((Wingdings, 0xCE), 0x1F654)
  ,((Wingdings, 0xCF), 0x1F657)
  ,((Wingdings, 0xD0), 0x1F656)
  ,((Wingdings, 0xD1), 0x1F650)
  ,((Wingdings, 0xD2), 0x1F651)
  ,((Wingdings, 0xD3), 0x1F652)
  ,((Wingdings, 0xD4), 0x1F653)
  ,((Wingdings, 0xD5), 0x232B)
  ,((Wingdings, 0xD6), 0x2326)
  ,((Wingdings, 0xD7), 0x2B98)
  ,((Wingdings, 0xD8), 0x2B9A)
  ,((Wingdings, 0xD9), 0x2B99)
  ,((Wingdings, 0xDA), 0x2B9B)
  ,((Wingdings, 0xDB), 0x2B88)
  ,((Wingdings, 0xDC), 0x2B8A)
  ,((Wingdings, 0xDD), 0x2B89)
  ,((Wingdings, 0xDE), 0x2B8B)
  ,((Wingdings, 0xDF), 0x1F868)
  ,((Wingdings, 0xE0), 0x1F86A)
  ,((Wingdings, 0xE1), 0x1F869)
  ,((Wingdings, 0xE2), 0x1F86B)
  ,((Wingdings, 0xE3), 0x1F86C)
  ,((Wingdings, 0xE4), 0x1F86D)
  ,((Wingdings, 0xE5), 0x1F86F)
  ,((Wingdings, 0xE6), 0x1F86E)
  ,((Wingdings, 0xE7), 0x1F878)
  ,((Wingdings, 0xE8), 0x1F87A)
  ,((Wingdings, 0xE9), 0x1F879)
  ,((Wingdings, 0xEA), 0x1F87B)
  ,((Wingdings, 0xEB), 0x1F87C)
  ,((Wingdings, 0xEC), 0x1F87D)
  ,((Wingdings, 0xED), 0x1F87F)
  ,((Wingdings, 0xEE), 0x1F87E)
  ,((Wingdings, 0xEF), 0x21E6)
  ,((Wingdings, 0xF0), 0x21E8)
  ,((Wingdings, 0xF1), 0x21E7)
  ,((Wingdings, 0xF2), 0x21E9)
  ,((Wingdings, 0xF3), 0x2B04)
  ,((Wingdings, 0xF4), 0x21F3)
  ,((Wingdings, 0xF5), 0x2B00)
  ,((Wingdings, 0xF6), 0x2B01)
  ,((Wingdings, 0xF7), 0x2B03)
  ,((Wingdings, 0xF8), 0x2B02)
  ,((Wingdings, 0xF9), 0x1F8AC)
  ,((Wingdings, 0xFA), 0x1F8AD)
  ,((Wingdings, 0xFB), 0x1F5F6)
  ,((Wingdings, 0xFC), 0x2714)
  ,((Wingdings, 0xFD), 0x1F5F7)
  ,((Wingdings, 0xFE), 0x1F5F9)
  ,((Wingdings2, 0x21), 0x1F58A)
  ,((Wingdings2, 0x22), 0x1F58B)
  ,((Wingdings2, 0x23), 0x1F58C)
  ,((Wingdings2, 0x24), 0x1F58D)
  ,((Wingdings2, 0x25), 0x2704)
  ,((Wingdings2, 0x26), 0x2700)
  ,((Wingdings2, 0x27), 0x1F57E)
  ,((Wingdings2, 0x28), 0x1F57D)
  ,((Wingdings2, 0x29), 0x1F5C5)
  ,((Wingdings2, 0x2A), 0x1F5C6)
  ,((Wingdings2, 0x2B), 0x1F5C7)
  ,((Wingdings2, 0x2C), 0x1F5C8)
  ,((Wingdings2, 0x2D), 0x1F5C9)
  ,((Wingdings2, 0x2E), 0x1F5CA)
  ,((Wingdings2, 0x2F), 0x1F5CB)
  ,((Wingdings2, 0x30), 0x1F5CC)
  ,((Wingdings2, 0x31), 0x1F5CD)
  ,((Wingdings2, 0x32), 0x1F4CB)
  ,((Wingdings2, 0x33), 0x1F5D1)
  ,((Wingdings2, 0x34), 0x1F5D4)
  ,((Wingdings2, 0x35), 0x1F5B5)
  ,((Wingdings2, 0x36), 0x1F5B6)
  ,((Wingdings2, 0x37), 0x1F5B7)
  ,((Wingdings2, 0x38), 0x1F5B8)
  ,((Wingdings2, 0x39), 0x1F5AD)
  ,((Wingdings2, 0x3A), 0x1F5AF)
  ,((Wingdings2, 0x3B), 0x1F5B1)
  ,((Wingdings2, 0x3C), 0x1F592)
  ,((Wingdings2, 0x3D), 0x1F593)
  ,((Wingdings2, 0x3E), 0x1F598)
  ,((Wingdings2, 0x3F), 0x1F599)
  ,((Wingdings2, 0x40), 0x1F59A)
  ,((Wingdings2, 0x41), 0x1F59B)
  ,((Wingdings2, 0x42), 0x1F448)
  ,((Wingdings2, 0x43), 0x1F449)
  ,((Wingdings2, 0x44), 0x1F59C)
  ,((Wingdings2, 0x45), 0x1F59D)
  ,((Wingdings2, 0x46), 0x1F59E)
  ,((Wingdings2, 0x47), 0x1F59F)
  ,((Wingdings2, 0x48), 0x1F5A0)
  ,((Wingdings2, 0x49), 0x1F5A1)
  ,((Wingdings2, 0x4A), 0x1F446)
  ,((Wingdings2, 0x4B), 0x1F447)
  ,((Wingdings2, 0x4C), 0x1F5A2)
  ,((Wingdings2, 0x4D), 0x1F5A3)
  ,((Wingdings2, 0x4E), 0x1F591)
  ,((Wingdings2, 0x4F), 0x1F5F4)
  ,((Wingdings2, 0x50), 0x2713)
  ,((Wingdings2, 0x51), 0x1F5F5)
  ,((Wingdings2, 0x52), 0x2611)
  ,((Wingdings2, 0x53), 0x2612)
  ,((Wingdings2, 0x54), 0x2612)
  ,((Wingdings2, 0x55), 0x2BBE)
  ,((Wingdings2, 0x56), 0x2BBF)
  ,((Wingdings2, 0x57), 0x29B8)
  ,((Wingdings2, 0x58), 0x29B8)
  ,((Wingdings2, 0x59), 0x1F671)
  ,((Wingdings2, 0x5A), 0x1F674)
  ,((Wingdings2, 0x5B), 0x1F672)
  ,((Wingdings2, 0x5C), 0x1F673)
  ,((Wingdings2, 0x5D), 0x203D)
  ,((Wingdings2, 0x5E), 0x1F679)
  ,((Wingdings2, 0x5F), 0x1F67A)
  ,((Wingdings2, 0x60), 0x1F67B)
  ,((Wingdings2, 0x61), 0x1F666)
  ,((Wingdings2, 0x62), 0x1F664)
  ,((Wingdings2, 0x63), 0x1F665)
  ,((Wingdings2, 0x64), 0x1F667)
  ,((Wingdings2, 0x65), 0x1F65A)
  ,((Wingdings2, 0x66), 0x1F658)
  ,((Wingdings2, 0x67), 0x1F659)
  ,((Wingdings2, 0x68), 0x1F65B)
  ,((Wingdings2, 0x69), 0x24EA)
  ,((Wingdings2, 0x6A), 0x2460)
  ,((Wingdings2, 0x6B), 0x2461)
  ,((Wingdings2, 0x6C), 0x2462)
  ,((Wingdings2, 0x6D), 0x2463)
  ,((Wingdings2, 0x6E), 0x2464)
  ,((Wingdings2, 0x6F), 0x2465)
  ,((Wingdings2, 0x70), 0x2466)
  ,((Wingdings2, 0x71), 0x2467)
  ,((Wingdings2, 0x72), 0x2468)
  ,((Wingdings2, 0x73), 0x2469)
  ,((Wingdings2, 0x74), 0x24FF)
  ,((Wingdings2, 0x75), 0x2776)
  ,((Wingdings2, 0x76), 0x2777)
  ,((Wingdings2, 0x77), 0x2778)
  ,((Wingdings2, 0x78), 0x2779)
  ,((Wingdings2, 0x79), 0x277A)
  ,((Wingdings2, 0x7A), 0x277B)
  ,((Wingdings2, 0x7B), 0x277C)
  ,((Wingdings2, 0x7C), 0x277D)
  ,((Wingdings2, 0x7D), 0x277E)
  ,((Wingdings2, 0x7E), 0x277F)
  ,((Wingdings2, 0x80), 0x2609)
  ,((Wingdings2, 0x81), 0x1F315)
  ,((Wingdings2, 0x82), 0x263D)
  ,((Wingdings2, 0x83), 0x263E)
  ,((Wingdings2, 0x84), 0x2E3F)
  ,((Wingdings2, 0x85), 0x271D)
  ,((Wingdings2, 0x86), 0x1F547)
  ,((Wingdings2, 0x87), 0x1F55C)
  ,((Wingdings2, 0x88), 0x1F55D)
  ,((Wingdings2, 0x89), 0x1F55E)
  ,((Wingdings2, 0x8A), 0x1F55F)
  ,((Wingdings2, 0x8B), 0x1F560)
  ,((Wingdings2, 0x8C), 0x1F561)
  ,((Wingdings2, 0x8D), 0x1F562)
  ,((Wingdings2, 0x8E), 0x1F563)
  ,((Wingdings2, 0x8F), 0x1F564)
  ,((Wingdings2, 0x90), 0x1F565)
  ,((Wingdings2, 0x91), 0x1F566)
  ,((Wingdings2, 0x92), 0x1F567)
  ,((Wingdings2, 0x93), 0x1F668)
  ,((Wingdings2, 0x94), 0x1F669)
  ,((Wingdings2, 0x95), 0x2022)
  ,((Wingdings2, 0x96), 0x25CF)
  ,((Wingdings2, 0x97), 0x26AB)
  ,((Wingdings2, 0x98), 0x2B24)
  ,((Wingdings2, 0x99), 0x1F785)
  ,((Wingdings2, 0x9A), 0x1F786)
  ,((Wingdings2, 0x9B), 0x1F787)
  ,((Wingdings2, 0x9C), 0x1F788)
  ,((Wingdings2, 0x9D), 0x1F78A)
  ,((Wingdings2, 0x9E), 0x29BF)
  ,((Wingdings2, 0x9F), 0x25FE)
  ,((Wingdings2, 0xA1), 0x25FC)
  ,((Wingdings2, 0xA2), 0x2B1B)
  ,((Wingdings2, 0xA3), 0x2B1C)
  ,((Wingdings2, 0xA4), 0x1F791)
  ,((Wingdings2, 0xA5), 0x1F792)
  ,((Wingdings2, 0xA6), 0x1F793)
  ,((Wingdings2, 0xA7), 0x1F794)
  ,((Wingdings2, 0xA8), 0x25A3)
  ,((Wingdings2, 0xA9), 0x1F795)
  ,((Wingdings2, 0xAA), 0x1F796)
  ,((Wingdings2, 0xAB), 0x1F797)
  ,((Wingdings2, 0xAC), 0x2B29)
  ,((Wingdings2, 0xAD), 0x2B25)
  ,((Wingdings2, 0xAE), 0x25C6)
  ,((Wingdings2, 0xAF), 0x25C7)
  ,((Wingdings2, 0xB0), 0x1F79A)
  ,((Wingdings2, 0xB1), 0x25C8)
  ,((Wingdings2, 0xB2), 0x1F79B)
  ,((Wingdings2, 0xB3), 0x1F79C)
  ,((Wingdings2, 0xB4), 0x1F79D)
  ,((Wingdings2, 0xB5), 0x2B2A)
  ,((Wingdings2, 0xB6), 0x2B27)
  ,((Wingdings2, 0xB7), 0x29EB)
  ,((Wingdings2, 0xB8), 0x25CA)
  ,((Wingdings2, 0xB9), 0x1F7A0)
  ,((Wingdings2, 0xBA), 0x25D6)
  ,((Wingdings2, 0xBB), 0x25D7)
  ,((Wingdings2, 0xBC), 0x2BCA)
  ,((Wingdings2, 0xBD), 0x2BCB)
  ,((Wingdings2, 0xBE), 0x25FC)
  ,((Wingdings2, 0xBF), 0x2B25)
  ,((Wingdings2, 0xC0), 0x2B1F)
  ,((Wingdings2, 0xC1), 0x2BC2)
  ,((Wingdings2, 0xC2), 0x2B23)
  ,((Wingdings2, 0xC3), 0x2B22)
  ,((Wingdings2, 0xC4), 0x2BC3)
  ,((Wingdings2, 0xC5), 0x2BC4)
  ,((Wingdings2, 0xC6), 0x1F7A1)
  ,((Wingdings2, 0xC7), 0x1F7A2)
  ,((Wingdings2, 0xC8), 0x1F7A3)
  ,((Wingdings2, 0xC9), 0x1F7A4)
  ,((Wingdings2, 0xCA), 0x1F7A5)
  ,((Wingdings2, 0xCB), 0x1F7A6)
  ,((Wingdings2, 0xCC), 0x1F7A7)
  ,((Wingdings2, 0xCD), 0x1F7A8)
  ,((Wingdings2, 0xCE), 0x1F7A9)
  ,((Wingdings2, 0xCF), 0x1F7AA)
  ,((Wingdings2, 0xD0), 0x1F7AB)
  ,((Wingdings2, 0xD1), 0x1F7AC)
  ,((Wingdings2, 0xD2), 0x1F7AD)
  ,((Wingdings2, 0xD3), 0x1F7AE)
  ,((Wingdings2, 0xD4), 0x1F7AF)
  ,((Wingdings2, 0xD5), 0x1F7B0)
  ,((Wingdings2, 0xD6), 0x1F7B1)
  ,((Wingdings2, 0xD7), 0x1F7B2)
  ,((Wingdings2, 0xD8), 0x1F7B3)
  ,((Wingdings2, 0xD9), 0x1F7B4)
  ,((Wingdings2, 0xDA), 0x1F7B5)
  ,((Wingdings2, 0xDB), 0x1F7B6)
  ,((Wingdings2, 0xDC), 0x1F7B7)
  ,((Wingdings2, 0xDD), 0x1F7B8)
  ,((Wingdings2, 0xDE), 0x1F7B9)
  ,((Wingdings2, 0xDF), 0x1F7BA)
  ,((Wingdings2, 0xE0), 0x1F7BB)
  ,((Wingdings2, 0xE1), 0x1F7BC)
  ,((Wingdings2, 0xE2), 0x1F7BD)
  ,((Wingdings2, 0xE3), 0x1F7BE)
  ,((Wingdings2, 0xE4), 0x1F7BF)
  ,((Wingdings2, 0xE5), 0x1F7C0)
  ,((Wingdings2, 0xE6), 0x1F7C2)
  ,((Wingdings2, 0xE7), 0x1F7C4)
  ,((Wingdings2, 0xE8), 0x2726)
  ,((Wingdings2, 0xE9), 0x1F7C9)
  ,((Wingdings2, 0xEA), 0x2605)
  ,((Wingdings2, 0xEB), 0x2736)
  ,((Wingdings2, 0xEC), 0x1F7CB)
  ,((Wingdings2, 0xED), 0x2737)
  ,((Wingdings2, 0xEE), 0x1F7CF)
  ,((Wingdings2, 0xEF), 0x1F7D2)
  ,((Wingdings2, 0xF0), 0x2739)
  ,((Wingdings2, 0xF1), 0x1F7C3)
  ,((Wingdings2, 0xF2), 0x1F7C7)
  ,((Wingdings2, 0xF3), 0x272F)
  ,((Wingdings2, 0xF4), 0x1F7CD)
  ,((Wingdings2, 0xF5), 0x1F7D4)
  ,((Wingdings2, 0xF6), 0x2BCC)
  ,((Wingdings2, 0xF7), 0x2BCD)
  ,((Wingdings2, 0xF8), 0x203B)
  ,((Wingdings2, 0xF9), 0x2042)
  ,((Wingdings3, 0x21), 0x2B60)
  ,((Wingdings3, 0x22), 0x2B62)
  ,((Wingdings3, 0x23), 0x2B61)
  ,((Wingdings3, 0x24), 0x2B63)
  ,((Wingdings3, 0x25), 0x2B66)
  ,((Wingdings3, 0x26), 0x2B67)
  ,((Wingdings3, 0x27), 0x2B69)
  ,((Wingdings3, 0x28), 0x2B68)
  ,((Wingdings3, 0x29), 0x2B70)
  ,((Wingdings3, 0x2A), 0x2B72)
  ,((Wingdings3, 0x2B), 0x2B71)
  ,((Wingdings3, 0x2C), 0x2B73)
  ,((Wingdings3, 0x2D), 0x2B76)
  ,((Wingdings3, 0x2E), 0x2B78)
  ,((Wingdings3, 0x2F), 0x2B7B)
  ,((Wingdings3, 0x30), 0x2B7D)
  ,((Wingdings3, 0x31), 0x2B64)
  ,((Wingdings3, 0x32), 0x2B65)
  ,((Wingdings3, 0x33), 0x2B6A)
  ,((Wingdings3, 0x34), 0x2B6C)
  ,((Wingdings3, 0x35), 0x2B6B)
  ,((Wingdings3, 0x36), 0x2B6D)
  ,((Wingdings3, 0x37), 0x2B4D)
  ,((Wingdings3, 0x38), 0x2BA0)
  ,((Wingdings3, 0x39), 0x2BA1)
  ,((Wingdings3, 0x3A), 0x2BA2)
  ,((Wingdings3, 0x3B), 0x2BA3)
  ,((Wingdings3, 0x3C), 0x2BA4)
  ,((Wingdings3, 0x3D), 0x2BA5)
  ,((Wingdings3, 0x3E), 0x2BA6)
  ,((Wingdings3, 0x3F), 0x2BA7)
  ,((Wingdings3, 0x40), 0x2B90)
  ,((Wingdings3, 0x41), 0x2B91)
  ,((Wingdings3, 0x42), 0x2B92)
  ,((Wingdings3, 0x43), 0x2B93)
  ,((Wingdings3, 0x44), 0x2B80)
  ,((Wingdings3, 0x45), 0x2B83)
  ,((Wingdings3, 0x46), 0x2B7E)
  ,((Wingdings3, 0x47), 0x2B7F)
  ,((Wingdings3, 0x48), 0x2B84)
  ,((Wingdings3, 0x49), 0x2B86)
  ,((Wingdings3, 0x4A), 0x2B85)
  ,((Wingdings3, 0x4B), 0x2B87)
  ,((Wingdings3, 0x4C), 0x2B8F)
  ,((Wingdings3, 0x4D), 0x2B8D)
  ,((Wingdings3, 0x4E), 0x2B8E)
  ,((Wingdings3, 0x4F), 0x2B8C)
  ,((Wingdings3, 0x50), 0x2B6E)
  ,((Wingdings3, 0x51), 0x2B6F)
  ,((Wingdings3, 0x52), 0x238B)
  ,((Wingdings3, 0x53), 0x2324)
  ,((Wingdings3, 0x54), 0x2303)
  ,((Wingdings3, 0x55), 0x2325)
  ,((Wingdings3, 0x56), 0x23B5)
  ,((Wingdings3, 0x57), 0x237D)
  ,((Wingdings3, 0x58), 0x21EA)
  ,((Wingdings3, 0x59), 0x2BB8)
  ,((Wingdings3, 0x5A), 0x1F8A0)
  ,((Wingdings3, 0x5B), 0x1F8A1)
  ,((Wingdings3, 0x5C), 0x1F8A2)
  ,((Wingdings3, 0x5D), 0x1F8A3)
  ,((Wingdings3, 0x5E), 0x1F8A4)
  ,((Wingdings3, 0x5F), 0x1F8A5)
  ,((Wingdings3, 0x60), 0x1F8A6)
  ,((Wingdings3, 0x61), 0x1F8A7)
  ,((Wingdings3, 0x62), 0x1F8A8)
  ,((Wingdings3, 0x63), 0x1F8A9)
  ,((Wingdings3, 0x64), 0x1F8AA)
  ,((Wingdings3, 0x65), 0x1F8AB)
  ,((Wingdings3, 0x66), 0x2190)
  ,((Wingdings3, 0x67), 0x2192)
  ,((Wingdings3, 0x68), 0x2191)
  ,((Wingdings3, 0x69), 0x2193)
  ,((Wingdings3, 0x6A), 0x2196)
  ,((Wingdings3, 0x6B), 0x2197)
  ,((Wingdings3, 0x6C), 0x2199)
  ,((Wingdings3, 0x6D), 0x2198)
  ,((Wingdings3, 0x6E), 0x1F858)
  ,((Wingdings3, 0x6F), 0x1F859)
  ,((Wingdings3, 0x70), 0x25B2)
  ,((Wingdings3, 0x71), 0x25BC)
  ,((Wingdings3, 0x72), 0x25B3)
  ,((Wingdings3, 0x73), 0x25BD)
  ,((Wingdings3, 0x74), 0x25C4)
  ,((Wingdings3, 0x75), 0x25BA)
  ,((Wingdings3, 0x76), 0x25C1)
  ,((Wingdings3, 0x77), 0x25B7)
  ,((Wingdings3, 0x78), 0x25E3)
  ,((Wingdings3, 0x79), 0x25E2)
  ,((Wingdings3, 0x7A), 0x25E4)
  ,((Wingdings3, 0x7B), 0x25E5)
  ,((Wingdings3, 0x7C), 0x1F780)
  ,((Wingdings3, 0x7D), 0x1F782)
  ,((Wingdings3, 0x7E), 0x1F781)
  ,((Wingdings3, 0x80), 0x1F783)
  ,((Wingdings3, 0x81), 0x25B2)
  ,((Wingdings3, 0x82), 0x25BC)
  ,((Wingdings3, 0x83), 0x25C0)
  ,((Wingdings3, 0x84), 0x25B6)
  ,((Wingdings3, 0x85), 0x2B9C)
  ,((Wingdings3, 0x86), 0x2B9E)
  ,((Wingdings3, 0x87), 0x2B9D)
  ,((Wingdings3, 0x88), 0x2B9F)
  ,((Wingdings3, 0x89), 0x1F810)
  ,((Wingdings3, 0x8A), 0x1F812)
  ,((Wingdings3, 0x8B), 0x1F811)
  ,((Wingdings3, 0x8C), 0x1F813)
  ,((Wingdings3, 0x8D), 0x1F814)
  ,((Wingdings3, 0x8E), 0x1F816)
  ,((Wingdings3, 0x8F), 0x1F815)
  ,((Wingdings3, 0x90), 0x1F817)
  ,((Wingdings3, 0x91), 0x1F818)
  ,((Wingdings3, 0x92), 0x1F81A)
  ,((Wingdings3, 0x93), 0x1F819)
  ,((Wingdings3, 0x94), 0x1F81B)
  ,((Wingdings3, 0x95), 0x1F81C)
  ,((Wingdings3, 0x96), 0x1F81E)
  ,((Wingdings3, 0x97), 0x1F81D)
  ,((Wingdings3, 0x98), 0x1F81F)
  ,((Wingdings3, 0x99), 0x1F800)
  ,((Wingdings3, 0x9A), 0x1F802)
  ,((Wingdings3, 0x9B), 0x1F801)
  ,((Wingdings3, 0x9C), 0x1F803)
  ,((Wingdings3, 0x9D), 0x1F804)
  ,((Wingdings3, 0x9E), 0x1F806)
  ,((Wingdings3, 0x9F), 0x1F805)
  ,((Wingdings3, 0xA1), 0x1F808)
  ,((Wingdings3, 0xA2), 0x1F80A)
  ,((Wingdings3, 0xA3), 0x1F809)
  ,((Wingdings3, 0xA4), 0x1F80B)
  ,((Wingdings3, 0xA5), 0x1F820)
  ,((Wingdings3, 0xA6), 0x1F822)
  ,((Wingdings3, 0xA7), 0x1F824)
  ,((Wingdings3, 0xA8), 0x1F826)
  ,((Wingdings3, 0xA9), 0x1F828)
  ,((Wingdings3, 0xAA), 0x1F82A)
  ,((Wingdings3, 0xAB), 0x1F82C)
  ,((Wingdings3, 0xAC), 0x1F89C)
  ,((Wingdings3, 0xAD), 0x1F89D)
  ,((Wingdings3, 0xAE), 0x1F89E)
  ,((Wingdings3, 0xAF), 0x1F89F)
  ,((Wingdings3, 0xB0), 0x1F82E)
  ,((Wingdings3, 0xB1), 0x1F830)
  ,((Wingdings3, 0xB2), 0x1F832)
  ,((Wingdings3, 0xB3), 0x1F834)
  ,((Wingdings3, 0xB4), 0x1F836)
  ,((Wingdings3, 0xB5), 0x1F838)
  ,((Wingdings3, 0xB6), 0x1F83A)
  ,((Wingdings3, 0xB7), 0x1F839)
  ,((Wingdings3, 0xB8), 0x1F83B)
  ,((Wingdings3, 0xB9), 0x1F898)
  ,((Wingdings3, 0xBA), 0x1F89A)
  ,((Wingdings3, 0xBB), 0x1F899)
  ,((Wingdings3, 0xBC), 0x1F89B)
  ,((Wingdings3, 0xBD), 0x1F83C)
  ,((Wingdings3, 0xBE), 0x1F83E)
  ,((Wingdings3, 0xBF), 0x1F83D)
  ,((Wingdings3, 0xC0), 0x1F83F)
  ,((Wingdings3, 0xC1), 0x1F840)
  ,((Wingdings3, 0xC2), 0x1F842)
  ,((Wingdings3, 0xC3), 0x1F841)
  ,((Wingdings3, 0xC4), 0x1F843)
  ,((Wingdings3, 0xC5), 0x1F844)
  ,((Wingdings3, 0xC6), 0x1F846)
  ,((Wingdings3, 0xC7), 0x1F845)
  ,((Wingdings3, 0xC8), 0x1F847)
  ,((Wingdings3, 0xC9), 0x2BA8)
  ,((Wingdings3, 0xCA), 0x2BA9)
  ,((Wingdings3, 0xCB), 0x2BAA)
  ,((Wingdings3, 0xCC), 0x2BAB)
  ,((Wingdings3, 0xCD), 0x2BAC)
  ,((Wingdings3, 0xCE), 0x2BAD)
  ,((Wingdings3, 0xCF), 0x2BAE)
  ,((Wingdings3, 0xD0), 0x2BAF)
  ,((Wingdings3, 0xD1), 0x1F860)
  ,((Wingdings3, 0xD2), 0x1F862)
  ,((Wingdings3, 0xD3), 0x1F861)
  ,((Wingdings3, 0xD4), 0x1F863)
  ,((Wingdings3, 0xD5), 0x1F864)
  ,((Wingdings3, 0xD6), 0x1F865)
  ,((Wingdings3, 0xD7), 0x1F867)
  ,((Wingdings3, 0xD8), 0x1F866)
  ,((Wingdings3, 0xD9), 0x1F870)
  ,((Wingdings3, 0xDA), 0x1F872)
  ,((Wingdings3, 0xDB), 0x1F871)
  ,((Wingdings3, 0xDC), 0x1F873)
  ,((Wingdings3, 0xDD), 0x1F874)
  ,((Wingdings3, 0xDE), 0x1F875)
  ,((Wingdings3, 0xDF), 0x1F877)
  ,((Wingdings3, 0xE0), 0x1F876)
  ,((Wingdings3, 0xE1), 0x1F880)
  ,((Wingdings3, 0xE2), 0x1F882)
  ,((Wingdings3, 0xE3), 0x1F881)
  ,((Wingdings3, 0xE4), 0x1F883)
  ,((Wingdings3, 0xE5), 0x1F884)
  ,((Wingdings3, 0xE6), 0x1F885)
  ,((Wingdings3, 0xE7), 0x1F887)
  ,((Wingdings3, 0xE8), 0x1F886)
  ,((Wingdings3, 0xE9), 0x1F890)
  ,((Wingdings3, 0xEA), 0x1F892)
  ,((Wingdings3, 0xEB), 0x1F891)
  ,((Wingdings3, 0xEC), 0x1F893)
  ,((Wingdings3, 0xED), 0x1F894)
  ,((Wingdings3, 0xEE), 0x1F896)
  ,((Wingdings3, 0xEF), 0x1F895)
  ,((Wingdings3, 0xF0), 0x1F897)
  ,((Webdings, 0x21), 0x1F577)
  ,((Webdings, 0x22), 0x1F578)
  ,((Webdings, 0x23), 0x1F572)
  ,((Webdings, 0x24), 0x1F576)
  ,((Webdings, 0x25), 0x1F3C6)
  ,((Webdings, 0x26), 0x1F396)
  ,((Webdings, 0x27), 0x1F587)
  ,((Webdings, 0x28), 0x1F5E8)
  ,((Webdings, 0x29), 0x1F5E9)
  ,((Webdings, 0x2A), 0x1F5F0)
  ,((Webdings, 0x2B), 0x1F5F1)
  ,((Webdings, 0x2C), 0x1F336)
  ,((Webdings, 0x2D), 0x1F397)
  ,((Webdings, 0x2E), 0x1F67E)
  ,((Webdings, 0x2F), 0x1F67C)
  ,((Webdings, 0x30), 0x1F5D5)
  ,((Webdings, 0x31), 0x1F5D6)
  ,((Webdings, 0x32), 0x1F5D7)
  ,((Webdings, 0x33), 0x23F4)
  ,((Webdings, 0x34), 0x23F5)
  ,((Webdings, 0x35), 0x23F6)
  ,((Webdings, 0x36), 0x23F7)
  ,((Webdings, 0x37), 0x23EA)
  ,((Webdings, 0x38), 0x23E9)
  ,((Webdings, 0x39), 0x23EE)
  ,((Webdings, 0x3A), 0x23ED)
  ,((Webdings, 0x3B), 0x23F8)
  ,((Webdings, 0x3C), 0x23F9)
  ,((Webdings, 0x3D), 0x23FA)
  ,((Webdings, 0x3E), 0x1F5DA)
  ,((Webdings, 0x3F), 0x1F5F3)
  ,((Webdings, 0x40), 0x1F6E0)
  ,((Webdings, 0x41), 0x1F3D7)
  ,((Webdings, 0x42), 0x1F3D8)
  ,((Webdings, 0x43), 0x1F3D9)
  ,((Webdings, 0x44), 0x1F3DA)
  ,((Webdings, 0x45), 0x1F3DC)
  ,((Webdings, 0x46), 0x1F3ED)
  ,((Webdings, 0x47), 0x1F3DB)
  ,((Webdings, 0x48), 0x1F3E0)
  ,((Webdings, 0x49), 0x1F3D6)
  ,((Webdings, 0x4A), 0x1F3DD)
  ,((Webdings, 0x4B), 0x1F6E3)
  ,((Webdings, 0x4C), 0x1F50D)
  ,((Webdings, 0x4D), 0x1F3D4)
  ,((Webdings, 0x4E), 0x1F441)
  ,((Webdings, 0x4F), 0x1F442)
  ,((Webdings, 0x50), 0x1F3DE)
  ,((Webdings, 0x51), 0x1F3D5)
  ,((Webdings, 0x52), 0x1F6E4)
  ,((Webdings, 0x53), 0x1F3DF)
  ,((Webdings, 0x54), 0x1F6F3)
  ,((Webdings, 0x55), 0x1F56C)
  ,((Webdings, 0x56), 0x1F56B)
  ,((Webdings, 0x57), 0x1F568)
  ,((Webdings, 0x58), 0x1F508)
  ,((Webdings, 0x59), 0x1F394)
  ,((Webdings, 0x5A), 0x1F395)
  ,((Webdings, 0x5B), 0x1F5EC)
  ,((Webdings, 0x5C), 0x1F67D)
  ,((Webdings, 0x5D), 0x1F5ED)
  ,((Webdings, 0x5E), 0x1F5EA)
  ,((Webdings, 0x5F), 0x1F5EB)
  ,((Webdings, 0x60), 0x2B94)
  ,((Webdings, 0x61), 0x2714)
  ,((Webdings, 0x62), 0x1F6B2)
  ,((Webdings, 0x63), 0x25A1)
  ,((Webdings, 0x64), 0x1F6E1)
  ,((Webdings, 0x65), 0x1F4E6)
  ,((Webdings, 0x66), 0x1F6F1)
  ,((Webdings, 0x67), 0x25A0)
  ,((Webdings, 0x68), 0x1F691)
  ,((Webdings, 0x69), 0x1F6C8)
  ,((Webdings, 0x6A), 0x1F6E9)
  ,((Webdings, 0x6B), 0x1F6F0)
  ,((Webdings, 0x6C), 0x1F7C8)
  ,((Webdings, 0x6D), 0x1F574)
  ,((Webdings, 0x6E), 0x26AB)
  ,((Webdings, 0x6F), 0x1F6E5)
  ,((Webdings, 0x70), 0x1F694)
  ,((Webdings, 0x71), 0x1F5D8)
  ,((Webdings, 0x72), 0x1F5D9)
  ,((Webdings, 0x73), 0x2753)
  ,((Webdings, 0x74), 0x1F6F2)
  ,((Webdings, 0x75), 0x1F687)
  ,((Webdings, 0x76), 0x1F68D)
  ,((Webdings, 0x77), 0x26F3)
  ,((Webdings, 0x78), 0x1F6C7)
  ,((Webdings, 0x79), 0x2296)
  ,((Webdings, 0x7A), 0x1F6AD)
  ,((Webdings, 0x7B), 0x1F5EE)
  ,((Webdings, 0x7C), 0x007C)
  ,((Webdings, 0x7D), 0x1F5EF)
  ,((Webdings, 0x7E), 0x1F5F2)
  ,((Webdings, 0x80), 0x1F6B9)
  ,((Webdings, 0x81), 0x1F6BA)
  ,((Webdings, 0x82), 0x1F6C9)
  ,((Webdings, 0x83), 0x1F6CA)
  ,((Webdings, 0x84), 0x1F6BC)
  ,((Webdings, 0x85), 0x1F47D)
  ,((Webdings, 0x86), 0x1F3CB)
  ,((Webdings, 0x87), 0x26F7)
  ,((Webdings, 0x88), 0x1F3C2)
  ,((Webdings, 0x89), 0x1F3CC)
  ,((Webdings, 0x8A), 0x1F3CA)
  ,((Webdings, 0x8B), 0x1F3C4)
  ,((Webdings, 0x8C), 0x1F3CD)
  ,((Webdings, 0x8D), 0x1F3CE)
  ,((Webdings, 0x8E), 0x1F698)
  ,((Webdings, 0x8F), 0x1F5E0)
  ,((Webdings, 0x90), 0x1F6E2)
  ,((Webdings, 0x91), 0x1F4B0)
  ,((Webdings, 0x92), 0x1F3F7)
  ,((Webdings, 0x93), 0x1F4B3)
  ,((Webdings, 0x94), 0x1F46A)
  ,((Webdings, 0x95), 0x1F5E1)
  ,((Webdings, 0x96), 0x1F5E2)
  ,((Webdings, 0x97), 0x1F5E3)
  ,((Webdings, 0x98), 0x272F)
  ,((Webdings, 0x99), 0x1F584)
  ,((Webdings, 0x9A), 0x1F585)
  ,((Webdings, 0x9B), 0x1F583)
  ,((Webdings, 0x9C), 0x1F586)
  ,((Webdings, 0x9D), 0x1F5B9)
  ,((Webdings, 0x9E), 0x1F5BA)
  ,((Webdings, 0x9F), 0x1F5BB)
  ,((Webdings, 0xA1), 0x1F570)
  ,((Webdings, 0xA2), 0x1F5BD)
  ,((Webdings, 0xA3), 0x1F5BE)
  ,((Webdings, 0xA4), 0x1F4CB)
  ,((Webdings, 0xA5), 0x1F5D2)
  ,((Webdings, 0xA6), 0x1F5D3)
  ,((Webdings, 0xA7), 0x1F4D6)
  ,((Webdings, 0xA8), 0x1F4DA)
  ,((Webdings, 0xA9), 0x1F5DE)
  ,((Webdings, 0xAA), 0x1F5DF)
  ,((Webdings, 0xAB), 0x1F5C3)
  ,((Webdings, 0xAC), 0x1F5C2)
  ,((Webdings, 0xAD), 0x1F5BC)
  ,((Webdings, 0xAE), 0x1F3AD)
  ,((Webdings, 0xAF), 0x1F39C)
  ,((Webdings, 0xB0), 0x1F398)
  ,((Webdings, 0xB1), 0x1F399)
  ,((Webdings, 0xB2), 0x1F3A7)
  ,((Webdings, 0xB3), 0x1F4BF)
  ,((Webdings, 0xB4), 0x1F39E)
  ,((Webdings, 0xB5), 0x1F4F7)
  ,((Webdings, 0xB6), 0x1F39F)
  ,((Webdings, 0xB7), 0x1F3AC)
  ,((Webdings, 0xB8), 0x1F4FD)
  ,((Webdings, 0xB9), 0x1F4F9)
  ,((Webdings, 0xBA), 0x1F4FE)
  ,((Webdings, 0xBB), 0x1F4FB)
  ,((Webdings, 0xBC), 0x1F39A)
  ,((Webdings, 0xBD), 0x1F39B)
  ,((Webdings, 0xBE), 0x1F4FA)
  ,((Webdings, 0xBF), 0x1F4BB)
  ,((Webdings, 0xC0), 0x1F5A5)
  ,((Webdings, 0xC1), 0x1F5A6)
  ,((Webdings, 0xC2), 0x1F5A7)
  ,((Webdings, 0xC3), 0x1F579)
  ,((Webdings, 0xC4), 0x1F3AE)
  ,((Webdings, 0xC5), 0x1F57B)
  ,((Webdings, 0xC6), 0x1F57C)
  ,((Webdings, 0xC7), 0x1F4DF)
  ,((Webdings, 0xC8), 0x1F581)
  ,((Webdings, 0xC9), 0x1F580)
  ,((Webdings, 0xCA), 0x1F5A8)
  ,((Webdings, 0xCB), 0x1F5A9)
  ,((Webdings, 0xCC), 0x1F5BF)
  ,((Webdings, 0xCD), 0x1F5AA)
  ,((Webdings, 0xCE), 0x1F5DC)
  ,((Webdings, 0xCF), 0x1F512)
  ,((Webdings, 0xD0), 0x1F513)
  ,((Webdings, 0xD1), 0x1F5DD)
  ,((Webdings, 0xD2), 0x1F4E5)
  ,((Webdings, 0xD3), 0x1F4E4)
  ,((Webdings, 0xD4), 0x1F573)
  ,((Webdings, 0xD5), 0x1F323)
  ,((Webdings, 0xD6), 0x1F324)
  ,((Webdings, 0xD7), 0x1F325)
  ,((Webdings, 0xD8), 0x1F326)
  ,((Webdings, 0xD9), 0x2601)
  ,((Webdings, 0xDA), 0x1F327)
  ,((Webdings, 0xDB), 0x1F328)
  ,((Webdings, 0xDC), 0x1F329)
  ,((Webdings, 0xDD), 0x1F32A)
  ,((Webdings, 0xDE), 0x1F32C)
  ,((Webdings, 0xDF), 0x1F32B)
  ,((Webdings, 0xE0), 0x1F31C)
  ,((Webdings, 0xE1), 0x1F321)
  ,((Webdings, 0xE2), 0x1F6CB)
  ,((Webdings, 0xE3), 0x1F6CF)
  ,((Webdings, 0xE4), 0x1F37D)
  ,((Webdings, 0xE5), 0x1F378)
  ,((Webdings, 0xE6), 0x1F6CE)
  ,((Webdings, 0xE7), 0x1F6CD)
  ,((Webdings, 0xE8), 0x24C5)
  ,((Webdings, 0xE9), 0x267F)
  ,((Webdings, 0xEA), 0x1F6C6)
  ,((Webdings, 0xEB), 0x1F588)
  ,((Webdings, 0xEC), 0x1F393)
  ,((Webdings, 0xED), 0x1F5E4)
  ,((Webdings, 0xEE), 0x1F5E5)
  ,((Webdings, 0xEF), 0x1F5E6)
  ,((Webdings, 0xF0), 0x1F5E7)
  ,((Webdings, 0xF1), 0x1F6EA)
  ,((Webdings, 0xF2), 0x1F43F)
  ,((Webdings, 0xF3), 0x1F426)
  ,((Webdings, 0xF4), 0x1F41F)
  ,((Webdings, 0xF5), 0x1F415)
  ,((Webdings, 0xF6), 0x1F408)
  ,((Webdings, 0xF7), 0x1F66C)
  ,((Webdings, 0xF8), 0x1F66E)
  ,((Webdings, 0xF9), 0x1F66D)
  ,((Webdings, 0xFA), 0x1F66F)
  ,((Webdings, 0xFB), 0x1F5FA)
  ,((Webdings, 0xFC), 0x1F30D)
  ,((Webdings, 0xFD), 0x1F30F)
  ,((Webdings, 0xFE), 0x1F30E)
  ,((Webdings, 0xFF), 0x1F54A)
  ,((Symbol,0x20), 0xA0)
  ,((Symbol,0x21), 0x21)
  ,((Symbol,0x22), 0x2200)
  ,((Symbol,0x23), 0x23)
  ,((Symbol,0x24), 0x2203)
  ,((Symbol,0x25), 0x25)
  ,((Symbol,0x26), 0x26)
  ,((Symbol,0x27), 0x220B)
  ,((Symbol,0x28), 0x28)
  ,((Symbol,0x29), 0x29)
  ,((Symbol,0x2A), 0x2217)
  ,((Symbol,0x2B), 0x2B)
  ,((Symbol,0x2C), 0x2C)
  ,((Symbol,0x2D), 0x2212)
  ,((Symbol,0x2E), 0x2E)
  ,((Symbol,0x2F), 0x2F)
  ,((Symbol,0x30), 0x30)
  ,((Symbol,0x31), 0x31)
  ,((Symbol,0x32), 0x32)
  ,((Symbol,0x33), 0x33)
  ,((Symbol,0x34), 0x34)
  ,((Symbol,0x35), 0x35)
  ,((Symbol,0x36), 0x36)
  ,((Symbol,0x37), 0x37)
  ,((Symbol,0x38), 0x38)
  ,((Symbol,0x39), 0x39)
  ,((Symbol,0x3A), 0x3A)
  ,((Symbol,0x3B), 0x3B)
  ,((Symbol,0x3C), 0x3C)
  ,((Symbol,0x3D), 0x3D)
  ,((Symbol,0x3E), 0x3E)
  ,((Symbol,0x3F), 0x3F)
  ,((Symbol,0x40), 0x2245)
  ,((Symbol,0x41), 0x391)
  ,((Symbol,0x42), 0x392)
  ,((Symbol,0x43), 0x3A7)
  ,((Symbol,0x44), 0x2206)
  ,((Symbol,0x45), 0x395)
  ,((Symbol,0x46), 0x3A6)
  ,((Symbol,0x47), 0x393)
  ,((Symbol,0x48), 0x397)
  ,((Symbol,0x49), 0x399)
  ,((Symbol,0x4A), 0x3D1)
  ,((Symbol,0x4B), 0x39A)
  ,((Symbol,0x4C), 0x39B)
  ,((Symbol,0x4D), 0x39C)
  ,((Symbol,0x4E), 0x39D)
  ,((Symbol,0x4F), 0x39F)
  ,((Symbol,0x50), 0x3A0)
  ,((Symbol,0x51), 0x398)
  ,((Symbol,0x52), 0x3A1)
  ,((Symbol,0x53), 0x3A3)
  ,((Symbol,0x54), 0x3A4)
  ,((Symbol,0x55), 0x3A5)
  ,((Symbol,0x56), 0x3C2)
  ,((Symbol,0x57), 0x2126)
  ,((Symbol,0x58), 0x39E)
  ,((Symbol,0x59), 0x3A8)
  ,((Symbol,0x5A), 0x396)
  ,((Symbol,0x5B), 0x5B)
  ,((Symbol,0x5C), 0x2234)
  ,((Symbol,0x5D), 0x5D)
  ,((Symbol,0x5E), 0x22A5)
  ,((Symbol,0x5F), 0x5F)
  ,((Symbol,0x60), 0xF8E5)
  ,((Symbol,0x61), 0x3B1)
  ,((Symbol,0x62), 0x3B2)
  ,((Symbol,0x63), 0x3C7)
  ,((Symbol,0x64), 0x3B4)
  ,((Symbol,0x65), 0x3B5)
  ,((Symbol,0x66), 0x3C6)
  ,((Symbol,0x67), 0x3B3)
  ,((Symbol,0x68), 0x3B7)
  ,((Symbol,0x69), 0x3B9)
  ,((Symbol,0x6A), 0x3D5)
  ,((Symbol,0x6B), 0x3BA)
  ,((Symbol,0x6C), 0x3BB)
  ,((Symbol,0x6D), 0x3BC)
  ,((Symbol,0x6E), 0x3BD)
  ,((Symbol,0x6F), 0x3BF)
  ,((Symbol,0x70), 0x3C0)
  ,((Symbol,0x71), 0x3B8)
  ,((Symbol,0x72), 0x3C1)
  ,((Symbol,0x73), 0x3C3)
  ,((Symbol,0x74), 0x3C4)
  ,((Symbol,0x75), 0x3C5)
  ,((Symbol,0x76), 0x3D6)
  ,((Symbol,0x77), 0x3C9)
  ,((Symbol,0x78), 0x3BE)
  ,((Symbol,0x79), 0x3C8)
  ,((Symbol,0x7A), 0x3B6)
  ,((Symbol,0x7B), 0x7B)
  ,((Symbol,0x7C), 0x7C)
  ,((Symbol,0x7D), 0x7D)
  ,((Symbol,0x7E), 0x223C)
  ,((Symbol,0xA0), 0x20AC)
  ,((Symbol,0xA1), 0x3D2)
  ,((Symbol,0xA2), 0x2032)
  ,((Symbol,0xA3), 0x2264)
  ,((Symbol,0xA4), 0x2215)
  ,((Symbol,0xA5), 0x221E)
  ,((Symbol,0xA6), 0x192)
  ,((Symbol,0xA7), 0x2663)
  ,((Symbol,0xA8), 0x2666)
  ,((Symbol,0xA9), 0x2665)
  ,((Symbol,0xAA), 0x2660)
  ,((Symbol,0xAB), 0x2194)
  ,((Symbol,0xAC), 0x2190)
  ,((Symbol,0xAD), 0x2191)
  ,((Symbol,0xAE), 0x2192)
  ,((Symbol,0xAF), 0x2193)
  ,((Symbol,0xB0), 0xB0)
  ,((Symbol,0xB1), 0xB1)
  ,((Symbol,0xB2), 0x2033)
  ,((Symbol,0xB3), 0x2265)
  ,((Symbol,0xB4), 0xD7)
  ,((Symbol,0xB5), 0x221D)
  ,((Symbol,0xB6), 0x2202)
  ,((Symbol,0xB7), 0x2022)
  ,((Symbol,0xB8), 0xF7)
  ,((Symbol,0xB9), 0x2260)
  ,((Symbol,0xBA), 0x2261)
  ,((Symbol,0xBB), 0x2248)
  ,((Symbol,0xBC), 0x2026)
  ,((Symbol,0xBD), 0xF8E6)
  ,((Symbol,0xBE), 0xF8E7)
  ,((Symbol,0xBF), 0x21B5)
  ,((Symbol,0xC0), 0x2135)
  ,((Symbol,0xC1), 0x2111)
  ,((Symbol,0xC2), 0x211C)
  ,((Symbol,0xC3), 0x2118)
  ,((Symbol,0xC4), 0x2297)
  ,((Symbol,0xC5), 0x2295)
  ,((Symbol,0xC6), 0x2205)
  ,((Symbol,0xC7), 0x2229)
  ,((Symbol,0xC8), 0x222A)
  ,((Symbol,0xC9), 0x2283)
  ,((Symbol,0xCA), 0x2287)
  ,((Symbol,0xCB), 0x2284)
  ,((Symbol,0xCC), 0x2282)
  ,((Symbol,0xCD), 0x2286)
  ,((Symbol,0xCE), 0x2208)
  ,((Symbol,0xCF), 0x2209)
  ,((Symbol,0xD0), 0x2220)
  ,((Symbol,0xD1), 0x2207)
  ,((Symbol,0xD2), 0xF6DA)
  ,((Symbol,0xD3), 0xF6D9)
  ,((Symbol,0xD4), 0xF6DB)
  ,((Symbol,0xD5), 0x220F)
  ,((Symbol,0xD6), 0x221A)
  ,((Symbol,0xD7), 0x22C5)
  ,((Symbol,0xD8), 0xAC)
  ,((Symbol,0xD9), 0x2227)
  ,((Symbol,0xDA), 0x2228)
  ,((Symbol,0xDB), 0x21D4)
  ,((Symbol,0xDC), 0x21D0)
  ,((Symbol,0xDD), 0x21D1)
  ,((Symbol,0xDE), 0x21D2)
  ,((Symbol,0xDF), 0x21D3)
  ,((Symbol,0xE0), 0x25CA)
  ,((Symbol,0xE1), 0x2329)
  ,((Symbol,0xE2), 0xF8E8)
  ,((Symbol,0xE3), 0xF8E9)
  ,((Symbol,0xE4), 0xF8EA)
  ,((Symbol,0xE5), 0x2211)
  ,((Symbol,0xE6), 0xF8EB)
  ,((Symbol,0xE7), 0xF8EC)
  ,((Symbol,0xE8), 0xF8ED)
  ,((Symbol,0xE9), 0xF8EE)
  ,((Symbol,0xEA), 0xF8EF)
  ,((Symbol,0xEB), 0xF8F0)
  ,((Symbol,0xEC), 0xF8F1)
  ,((Symbol,0xED), 0xF8F2)
  ,((Symbol,0xEE), 0xF8F3)
  ,((Symbol,0xEF), 0xF8F4)
  ,((Symbol,0xF1), 0x232A)
  ,((Symbol,0xF2), 0x222B)
  ,((Symbol,0xF3), 0x2320)
  ,((Symbol,0xF4), 0xF8F5)
  ,((Symbol,0xF5), 0x2321)
  ,((Symbol,0xF6), 0xF8F6)
  ,((Symbol,0xF7), 0xF8F7)
  ,((Symbol,0xF8), 0xF8F8)
  ,((Symbol,0xF9), 0xF8F9)
  ,((Symbol,0xFA), 0xF8FA)
  ,((Symbol,0xFB), 0xF8FB)
  ,((Symbol,0xFC), 0xF8FC)
  ,((Symbol,0xFD), 0xF8FD)
  ,((Symbol,0xFE), 0xF8FE)
  ]
