/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 * (c) ZeroTier, Inc.
 * https://www.zerotier.com/
 */

#ifndef ZT_SHAREDPTR_HPP
#define ZT_SHAREDPTR_HPP

#include "AtomicCounter.hpp"
#include "Mutex.hpp"

namespace ZeroTier {

/**
 * Simple zero-overhead introspective reference counted pointer
 *
 * This is an introspective shared pointer. Classes that need to be reference
 * counted must list this as a 'friend' and must have a private instance of
 * AtomicCounter called __refCount.
 */
template <typename T> class SharedPtr {
  public:
	SharedPtr() : _ptr((T*)0)
	{
	}
	SharedPtr(T* obj) : _ptr(obj)
	{
		++obj->__refCount;
	}
	SharedPtr(const SharedPtr& sp) : _ptr(sp._getAndInc())
	{
	}

	~SharedPtr()
	{
		if (_ptr) {
			if (--_ptr->__refCount <= 0) {
				delete _ptr;
			}
		}
	}

	inline SharedPtr& operator=(const SharedPtr& sp)
	{
		if (_ptr != sp._ptr) {
			T* p = sp._getAndInc();
			if (_ptr) {
				if (--_ptr->__refCount <= 0) {
					delete _ptr;
				}
			}
			_ptr = p;
		}
		return *this;
	}

	/**
	 * Set to a naked pointer and increment its reference count
	 *
	 * This assumes this SharedPtr is NULL and that ptr is not a 'zombie.' No
	 * checks are performed.
	 *
	 * @param ptr Naked pointer to assign
	 */
	inline void set(T* ptr)
	{
		zero();
		++ptr->__refCount;
		_ptr = ptr;
	}

	/**
	 * Swap with another pointer 'for free' without ref count overhead
	 *
	 * @param with Pointer to swap with
	 */
	inline void swap(SharedPtr& with)
	{
		T* tmp = _ptr;
		_ptr = with._ptr;
		with._ptr = tmp;
	}

	inline operator bool() const
	{
		return (_ptr != (T*)0);
	}
	inline T& operator*() const
	{
		return *_ptr;
	}
	inline T* operator->() const
	{
		return _ptr;
	}

	/**
	 * @return Raw pointer to held object
	 */
	inline T* ptr() const
	{
		return _ptr;
	}

	/**
	 * Set this pointer to NULL
	 */
	inline void zero()
	{
		if (_ptr) {
			if (--_ptr->__refCount <= 0) {
				delete _ptr;
			}
			_ptr = (T*)0;
		}
	}

	/**
	 * @return Number of references according to this object's ref count or 0 if NULL
	 */
	inline int references()
	{
		if (_ptr) {
			return _ptr->__refCount.load();
		}
		return 0;
	}

	inline bool operator==(const SharedPtr& sp) const
	{
		return (_ptr == sp._ptr);
	}
	inline bool operator!=(const SharedPtr& sp) const
	{
		return (_ptr != sp._ptr);
	}
	inline bool operator>(const SharedPtr& sp) const
	{
		return (_ptr > sp._ptr);
	}
	inline bool operator<(const SharedPtr& sp) const
	{
		return (_ptr < sp._ptr);
	}
	inline bool operator>=(const SharedPtr& sp) const
	{
		return (_ptr >= sp._ptr);
	}
	inline bool operator<=(const SharedPtr& sp) const
	{
		return (_ptr <= sp._ptr);
	}

  private:
	inline T* _getAndInc() const
	{
		if (_ptr) {
			++_ptr->__refCount;
		}
		return _ptr;
	}
	T* _ptr;
};

}	// namespace ZeroTier

#endif
