/*
 *  Copyright (C) 2021 - This file is part of libecc project
 *
 *  Authors:
 *      Ryad BENADJILA <ryadbenadjila@gmail.com>
 *      Arnaud EBALARD <arnaud.ebalard@ssi.gouv.fr>
 *
 *  This software is licensed under a dual BSD and GPL v2 license.
 *  See LICENSE file at the root folder of the project.
 */
#include "dsa.h"


/* We include the rand external dependency because we have to generate
 * some random data for the nonces.
 */
#include <libecc/external_deps/rand.h>
/* We include the printf external dependency for printf output */
#include <libecc/external_deps/print.h>
/* We include our common helpers */
#include "../common/common.h"

/*
 * The purpose of this example is to implement the DSA
 * related algorithms as per FIPS 186-4 based on libecc arithmetic
 * primitives.
 *
 * XXX: Please be aware that libecc has been designed for Elliptic
 * Curve cryptography, and as so the arithmetic primitives are
 * not optimized for big numbers >= 1024 bits usually used for DSA.
 * Additionnaly, a hard limit of our NN values makes it impossible
 * to exceed ~5300 bits in the best case (words of size 64 bits).
 *
 * All in all, please see this as a proof of concept of implementing
 * FIPS 186-4 rather than a production code. Use it at your own risk!
 *
 * !! DISCLAIMER !!
 * ================
 *
 * Althoug some efforts have been made to secure this implementation
 * of DSA (e.g. by protecting the private key and nonces using constant
 * time and blinding WHEN activated with BLINDING=1), please consider this
 * code as a proof of concept and use it at your own risk.
 *
 * All-in-all, this piece of code can be useful in some contexts, or risky to
 * use in other sensitive ones where advanced side-channels or fault attacks
 * have to be considered. Use this DSA code knowingly and at your own risk!
 *
 */


/* Import a DSA private key from buffers */
int dsa_import_priv_key(dsa_priv_key *priv, const u8 *p, u16 plen,
			const u8 *q, u16 qlen,
			const u8 *g, u16 glen,
			const u8 *x, u16 xlen)
{
	int ret, cmp;

	/* Sanity checks */
	MUST_HAVE((priv != NULL), ret, err);
	MUST_HAVE((p != NULL) && (q != NULL) && (g != NULL) && (x != NULL), ret, err);

	/* Import our big numbers */
	ret = _os2ip(&(priv->p), p, plen); EG(ret, err);
	ret = _os2ip(&(priv->q), q, qlen); EG(ret, err);
	ret = _os2ip(&(priv->g), g, glen); EG(ret, err);
	ret = _os2ip(&(priv->x), x, xlen); EG(ret, err);

	/* Sanity check that q < p */
	ret = nn_cmp(&(priv->q), &(priv->p), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);
	/* Sanity check that g < p */
	ret = nn_cmp(&(priv->g), &(priv->p), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);
	/* Sanity check that x < q */
	ret = nn_cmp(&(priv->x), &(priv->q), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);

err:
	if(ret && (priv != NULL)){
		IGNORE_RET_VAL(local_memset(priv, 0, sizeof(dsa_priv_key)));
	}

	return ret;
}

/* Import a DSA public key from buffers */
int dsa_import_pub_key(dsa_pub_key *pub, const u8 *p, u16 plen,
			const u8 *q, u16 qlen,
			const u8 *g, u16 glen,
			const u8 *y, u16 ylen)
{
	int ret, cmp;

	/* Sanity checks */
	MUST_HAVE((pub != NULL), ret, err);
	MUST_HAVE((p != NULL) && (q != NULL) && (g != NULL) && (y != NULL), ret, err);

	/* Import our big numbers */
	ret = _os2ip(&(pub->p), p, plen); EG(ret, err);
	ret = _os2ip(&(pub->q), q, qlen); EG(ret, err);
	ret = _os2ip(&(pub->g), g, glen); EG(ret, err);
	ret = _os2ip(&(pub->y), y, ylen); EG(ret, err);

	/* Sanity check that q < p */
	ret = nn_cmp(&(pub->q), &(pub->p), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);
	/* Sanity check that g < p */
	ret = nn_cmp(&(pub->g), &(pub->p), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);
	/* Sanity check that y < p */
	ret = nn_cmp(&(pub->y), &(pub->p), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);

err:
	if(ret && (pub != NULL)){
		IGNORE_RET_VAL(local_memset(pub, 0, sizeof(dsa_pub_key)));
	}

	return ret;
}



/* Compute a DSA public key from a private key.
 * The public key is computed using modular exponentiation of the generator
 * with the private key.
 */
int dsa_compute_pub_from_priv(dsa_pub_key *pub, const dsa_priv_key *priv)
{
	int ret, cmp;
	nn_src_t p, q, g, x;
	nn x_;
	x_.magic = WORD(0);

	MUST_HAVE((pub != NULL) && (priv != NULL), ret, err);

	/* Make things more readable */
	p = &(priv->p);
	q = &(priv->q);
	g = &(priv->g);
	x = &(priv->x);

	/* Sanity checks */
	ret = nn_check_initialized(p); EG(ret, err);
	ret = nn_check_initialized(q); EG(ret, err);
	ret = nn_check_initialized(g); EG(ret, err);
	ret = nn_check_initialized(x); EG(ret, err);

	/* Sanity check that q < p */
	ret = nn_cmp(&(priv->q), &(priv->p), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);
	/* Sanity check that g < p */
	ret = nn_cmp(&(priv->g), &(priv->p), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);
	/* Sanity check that x < q */
	ret = nn_cmp(&(priv->x), &(priv->q), &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);

	ret = nn_init(&x_, 0); EG(ret, err);
	/* Blind the private key in all cases as this is a
	 * sensitive value.
	 */
	ret = _blind_scalar(x, q, &x_); EG(ret, err);
	ret = _fix_scalar_msb(&x_, q, &x_); EG(ret, err);

	/* Perform the exponentiation y = g**x mod (p) */
	ret = nn_mod_pow(&(pub->y), g, &x_, p); EG(ret, err);

	/* Initialize the public key */
	ret = nn_copy(&(pub->p), p); EG(ret, err);
	ret = nn_copy(&(pub->q), q); EG(ret, err);
	ret = nn_copy(&(pub->g), g);

err:
	if(ret && (pub != NULL)){
		IGNORE_RET_VAL(local_memset(pub, 0, sizeof(dsa_pub_key)));
	}

	nn_uninit(&x_);

	PTR_NULLIFY(p);
	PTR_NULLIFY(q);
	PTR_NULLIFY(g);
	PTR_NULLIFY(x);

	return ret;
}

/* Generate a DSA signature
 * This implements "DSA Signature Generation" in section 4.6 of FIPS 186-4.
 */
int dsa_sign(const dsa_priv_key *priv, const u8 *msg, u32 msglen,
	     const u8 *nonce, u16 noncelen,
	     u8 *sig, u16 siglen, gen_hash_alg_type dsa_hash)
{
	int ret, iszero;
	/* N is the bit length of q */
	bitcnt_t N, rshift;
	/* Length of the hash function */
	u8 hlen, block_size;
	nn_src_t p, q, g, x;
	/* The nonce, it inverse and its protected version */
	nn k, kinv, k_;
	/* r, s and z */
	nn r, s, z;
	/* Hash */
	u8 hash[MAX_DIGEST_SIZE];
#ifdef USE_SIG_BLINDING
	/* b is the blinding mask */
	nn b;
	b.magic = WORD(0);
#endif /* USE_SIG_BLINDING */
	k.magic = kinv.magic = k_.magic = r.magic = s.magic = z.magic = WORD(0);

	/* Sanity checks */
	MUST_HAVE((priv != NULL) && (msg != NULL) && (sig != NULL), ret, err);

	ret = local_memset(hash, 0, sizeof(hash)); EG(ret, err);

	/* Make things more readable */
	p = &(priv->p);
	q = &(priv->q);
	g = &(priv->g);
	x = &(priv->x);

	/* Sanity checks */
	ret = nn_check_initialized(p); EG(ret, err);
	ret = nn_check_initialized(q); EG(ret, err);
	ret = nn_check_initialized(g); EG(ret, err);
	ret = nn_check_initialized(x); EG(ret, err);

	/* Let N be the bit length of q. Let min(N, outlen) denote the minimum
	 * of the positive integers N and outlen, where outlen is the bit length
	 * of the hash function output block.
	 */
	ret = nn_bitlen(q, &N); EG(ret, err);
	ret = gen_hash_get_hash_sizes(dsa_hash, &hlen, &block_size); EG(ret, err);
	MUST_HAVE((hlen <= MAX_DIGEST_SIZE), ret, err);

	/* Sanity check on the signature length */
	MUST_HAVE((siglen == (2 * BYTECEIL(N))), ret, err);

restart:
	/* If the nonce is imposed, use it. Else get a random modulo q */
	if(nonce != NULL){
		ret = _os2ip(&k, nonce, noncelen); EG(ret, err);
	}
	else{
		ret = nn_get_random_mod(&k, q); EG(ret, err);
	}

	/* Fix the MSB of our scalar */
	ret = nn_copy(&k_, &k); EG(ret, err);
#ifdef USE_SIG_BLINDING
	/* Blind the scalar */
	ret = _blind_scalar(&k_, q, &k_); EG(ret, err);
#endif /* USE_SIG_BLINDING */
	ret = _fix_scalar_msb(&k_, q, &k_); EG(ret, err);
	/* r = (g**k mod p) mod q */
	ret = nn_init(&r, 0); EG(ret, err);
	/* Exponentiation modulo p */
	ret = nn_mod_pow(&r, g, &k_, p); EG(ret, err);
	/* Modulo q */
	ret = nn_mod(&r, &r, q); EG(ret, err);

	/* If r is 0, restart the process */
	ret = nn_iszero(&r, &iszero); EG(ret, err);
	if (iszero) {
		goto restart;
 	}

	/* Export r */
	ret = _i2osp(&r, sig, (siglen / 2)); EG(ret, err);

	/* Compute the hash */
	ret = gen_hash_hfunc(msg, msglen, hash, dsa_hash); EG(ret, err);
	/* z = the leftmost min(N, outlen) bits of Hash(M) */
        rshift = 0;
        if ((hlen * 8) > N) {
                rshift = (bitcnt_t)((hlen * 8) - N);
        }
	ret = _os2ip(&z, hash, hlen); EG(ret, err);
	if (rshift) {
		ret = nn_rshift_fixedlen(&z, &z, rshift); EG(ret, err);
	}
	ret = nn_mod(&z, &z, q); EG(ret, err);

#ifdef USE_SIG_BLINDING
	/* Note: if we use blinding, r and e are multiplied by
	 * a random value b in ]0,q[ */
	ret = nn_get_random_mod(&b, q); EG(ret, err);
        /* Blind r with b */
        ret = nn_mod_mul(&r, &r, &b, q); EG(ret, err);
        /* Blind the message z */
        ret = nn_mod_mul(&z, &z, &b, q); EG(ret, err);
        /*
         * In case of blinding, we compute (b*k)^-1, and b^-1 will
         * automatically unblind (r*x) in the following.
         */
        ret = nn_mod_mul(&k, &k, &b, q); EG(ret, err);
#endif /* USE_SIG_BLINDING */

	/* Compute s = k^-1 * (xr + z) mod q  */
        /* Compute k^-1 mod q */
        /* NOTE: we use Fermat's little theorem inversion for
         * constant time here. This is possible since q is prime.
         */
        ret = nn_modinv_fermat(&kinv, &k, q); EG(ret, err);
	ret = nn_mod_mul(&s, &r, x, q); EG(ret, err);
	ret = nn_mod_add(&s, &s, &z, q); EG(ret, err);
	ret = nn_mod_mul(&s, &kinv, &s, q); EG(ret, err);

	/* If s is 0, restart the process */
	ret = nn_iszero(&s, &iszero); EG(ret, err);
	if (iszero) {
		goto restart;
 	}

	/* Export s */
	ret = _i2osp(&s, sig + (siglen / 2), (siglen / 2));

err:
	if(ret && (sig != NULL)){
		IGNORE_RET_VAL(local_memset(sig, 0, siglen));
	}

	nn_uninit(&k);
	nn_uninit(&kinv);
	nn_uninit(&k_);
#ifdef USE_SIG_BLINDING
	nn_uninit(&b);
#endif
	nn_uninit(&r);
	nn_uninit(&s);
	nn_uninit(&z);

	PTR_NULLIFY(p);
	PTR_NULLIFY(q);
	PTR_NULLIFY(g);
	PTR_NULLIFY(x);

	return ret;
}



/* Verify a DSA signature
 * This implements "DSA Signature Verification and Validation" in section 4.7 of FIPS 186-4.
 */
int dsa_verify(const dsa_pub_key *pub, const u8 *msg, u32 msglen,
	     const u8 *sig, u16 siglen, gen_hash_alg_type dsa_hash)
{
	int ret, iszero, cmp;
	/* N is the bit length of q */
	bitcnt_t N, rshift;
	/* Length of the hash function */
	u8 hlen, block_size;
	nn_src_t p, q, g, y;
	/* r, s */
	nn r, s, z;
	/* u1, u2, and v */
	nn u1, u2, v;
	/* Hash */
	u8 hash[MAX_DIGEST_SIZE];
	r.magic = s.magic = z.magic = u1.magic = u2.magic = WORD(0);

	/* Sanity checks */
	MUST_HAVE((pub != NULL) && (msg != NULL) && (sig != NULL), ret, err);

	ret = local_memset(hash, 0, sizeof(hash)); EG(ret, err);

	/* Make things more readable */
	p = &(pub->p);
	q = &(pub->q);
	g = &(pub->g);
	y = &(pub->y);

	/* Sanity checks */
	ret = nn_check_initialized(p); EG(ret, err);
	ret = nn_check_initialized(q); EG(ret, err);
	ret = nn_check_initialized(g); EG(ret, err);
	ret = nn_check_initialized(y); EG(ret, err);

	/* Sanity check on the signature length */
	ret = nn_bitlen(q, &N); EG(ret, err);
	MUST_HAVE((siglen == (2 * BYTECEIL(N))), ret, err);

	/* Extract r and s */
	ret = _os2ip(&r, sig, (siglen / 2)); EG(ret, err);
	ret = _os2ip(&s, sig + (siglen / 2), (siglen / 2)); EG(ret, err);

	/* Return an error if r = 0 or s = 0 */
	ret = nn_iszero(&r, &iszero); EG(ret, err);
	MUST_HAVE((!iszero), ret, err);
	ret = nn_iszero(&s, &iszero); EG(ret, err);
	MUST_HAVE((!iszero), ret, err);
	/* Check thatt 0 < r′ < q and 0 < s′ < q */
	ret = nn_cmp(&r, q, &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);
	ret = nn_cmp(&s, q, &cmp); EG(ret, err);
	MUST_HAVE((cmp < 0), ret, err);

	/* Compute the hash */
	ret = gen_hash_get_hash_sizes(dsa_hash, &hlen, &block_size); EG(ret, err);
	MUST_HAVE((hlen <= MAX_DIGEST_SIZE), ret, err);
	ret = gen_hash_hfunc(msg, msglen, hash, dsa_hash); EG(ret, err);
	/* z = the leftmost min(N, outlen) bits of Hash(M) */
        rshift = 0;
        if ((hlen * 8) > N) {
                rshift = (bitcnt_t)((hlen * 8) - N);
        }
	ret = _os2ip(&z, hash, hlen); EG(ret, err);
	if (rshift) {
		ret = nn_rshift_fixedlen(&z, &z, rshift); EG(ret, err);
	}
	ret = nn_mod(&z, &z, q); EG(ret, err);

	/* Compute w = s**-1 mod (q) in s */
        ret = nn_modinv(&s, &s, q); EG(ret, err);

	/* u1 = (zw) mod q */
	ret = nn_mod_mul(&u1, &z, &s, q); EG(ret, err);
	/* u2 = ((r′)w) mod q */
	ret = nn_mod_mul(&u2, &r, &s, q); EG(ret, err);
	/* Now compute v = = ((g**u1 y**u2) mod p) mod q */
	/* NOTE: no need to use a secure exponentiation here as we only
	 * manipulate public data.
	 */
	ret = _nn_mod_pow_insecure(&v, g, &u1, p); EG(ret, err);
	ret = _nn_mod_pow_insecure(&s, y, &u2, p); EG(ret, err);
	ret = nn_mod_mul(&v, &v, &s, p); EG(ret, err);
	ret = nn_mod(&v, &v, q); EG(ret, err);

	/* Check that v = r */
	ret = nn_cmp(&v, &r, &cmp); EG(ret, err);
	ret = (cmp != 0) ? -1 : 0;

err:
	nn_uninit(&r);
	nn_uninit(&s);
	nn_uninit(&z);
	nn_uninit(&u1);
	nn_uninit(&u2);
	nn_uninit(&v);

	PTR_NULLIFY(p);
	PTR_NULLIFY(q);
	PTR_NULLIFY(g);
	PTR_NULLIFY(y);

	return ret;
}

#ifdef DSA
#include <libecc/utils/print_buf.h>
int main(int argc, char *argv[])
{
 	int ret = 0;

	const u8 p[] = {
		0x90, 0x06, 0x64, 0x55, 0xB5, 0xCF, 0xC3, 0x8F, 0x9C, 0xAA, 0x4A, 0x48, 0xB4, 0x28, 0x1F, 0x29, 0x2C, 0x26, 0x0F, 0xEE, 0xF0, 0x1F, 0xD6, 0x10, 0x37, 0xE5, 0x62, 0x58,
		0xA7, 0x79, 0x5A, 0x1C, 0x7A, 0xD4, 0x60, 0x76, 0x98, 0x2C, 0xE6, 0xBB, 0x95, 0x69, 0x36, 0xC6, 0xAB, 0x4D, 0xCF, 0xE0, 0x5E, 0x67, 0x84, 0x58, 0x69, 0x40, 0xCA, 0x54,
		0x4B, 0x9B, 0x21, 0x40, 0xE1, 0xEB, 0x52, 0x3F, 0x00, 0x9D, 0x20, 0xA7, 0xE7, 0x88, 0x0E, 0x4E, 0x5B, 0xFA, 0x69, 0x0F, 0x1B, 0x90, 0x04, 0xA2, 0x78, 0x11, 0xCD, 0x99,
		0x04, 0xAF, 0x70, 0x42, 0x0E, 0xEF, 0xD6, 0xEA, 0x11, 0xEF, 0x7D, 0xA1, 0x29, 0xF5, 0x88, 0x35, 0xFF, 0x56, 0xB8, 0x9F, 0xAA, 0x63, 0x7B, 0xC9, 0xAC, 0x2E, 0xFA, 0xAB,
		0x90, 0x34, 0x02, 0x22, 0x9F, 0x49, 0x1D, 0x8D, 0x34, 0x85, 0x26, 0x1C, 0xD0, 0x68, 0x69, 0x9B, 0x6B, 0xA5, 0x8A, 0x1D, 0xDB, 0xBE, 0xF6, 0xDB, 0x51, 0xE8, 0xFE, 0x34,
		0xE8, 0xA7, 0x8E, 0x54, 0x2D, 0x7B, 0xA3, 0x51, 0xC2, 0x1E, 0xA8, 0xD8, 0xF1, 0xD2, 0x9F, 0x5D, 0x5D, 0x15, 0x93, 0x94, 0x87, 0xE2, 0x7F, 0x44, 0x16, 0xB0, 0xCA, 0x63,
		0x2C, 0x59, 0xEF, 0xD1, 0xB1, 0xEB, 0x66, 0x51, 0x1A, 0x5A, 0x0F, 0xBF, 0x61, 0x5B, 0x76, 0x6C, 0x58, 0x62, 0xD0, 0xBD, 0x8A, 0x3F, 0xE7, 0xA0, 0xE0, 0xDA, 0x0F, 0xB2,
		0xFE, 0x1F, 0xCB, 0x19, 0xE8, 0xF9, 0x99, 0x6A, 0x8E, 0xA0, 0xFC, 0xCD, 0xE5, 0x38, 0x17, 0x52, 0x38, 0xFC, 0x8B, 0x0E, 0xE6, 0xF2, 0x9A, 0xF7, 0xF6, 0x42, 0x77, 0x3E,
		0xBE, 0x8C, 0xD5, 0x40, 0x24, 0x15, 0xA0, 0x14, 0x51, 0xA8, 0x40, 0x47, 0x6B, 0x2F, 0xCE, 0xB0, 0xE3, 0x88, 0xD3, 0x0D, 0x4B, 0x37, 0x6C, 0x37, 0xFE, 0x40, 0x1C, 0x2A,
		0x2C, 0x2F, 0x94, 0x1D, 0xAD, 0x17, 0x9C, 0x54, 0x0C, 0x1C, 0x8C, 0xE0, 0x30, 0xD4, 0x60, 0xC4, 0xD9, 0x83, 0xBE, 0x9A, 0xB0, 0xB2, 0x0F, 0x69, 0x14, 0x4C, 0x1A, 0xE1,
		0x3F, 0x93, 0x83, 0xEA, 0x1C, 0x08, 0x50, 0x4F, 0xB0, 0xBF, 0x32, 0x15, 0x03, 0xEF, 0xE4, 0x34, 0x88, 0x31, 0x0D, 0xD8, 0xDC, 0x77, 0xEC, 0x5B, 0x83, 0x49, 0xB8, 0xBF,
		0xE9, 0x7C, 0x2C, 0x56, 0x0E, 0xA8, 0x78, 0xDE, 0x87, 0xC1, 0x1E, 0x3D, 0x59, 0x7F, 0x1F, 0xEA, 0x74, 0x2D, 0x73, 0xEE, 0xC7, 0xF3, 0x7B, 0xE4, 0x39, 0x49, 0xEF, 0x1A,
		0x0D, 0x15, 0xC3, 0xF3, 0xE3, 0xFC, 0x0A, 0x83, 0x35, 0x61, 0x70, 0x55, 0xAC, 0x91, 0x32, 0x8E, 0xC2, 0x2B, 0x50, 0xFC, 0x15, 0xB9, 0x41, 0xD3, 0xD1, 0x62, 0x4C, 0xD8,
		0x8B, 0xC2, 0x5F, 0x3E, 0x94, 0x1F, 0xDD, 0xC6, 0x20, 0x06, 0x89, 0x58, 0x1B, 0xFE, 0xC4, 0x16, 0xB4, 0xB2, 0xCB, 0x73,
	};

	const u8 q[] = {
		0xCF, 0xA0, 0x47, 0x8A, 0x54, 0x71, 0x7B, 0x08, 0xCE, 0x64, 0x80, 0x5B, 0x76, 0xE5, 0xB1, 0x42, 0x49, 0xA7, 0x7A, 0x48, 0x38, 0x46, 0x9D, 0xF7, 0xF7, 0xDC, 0x98, 0x7E,
		0xFC, 0xCF, 0xB1, 0x1D,
	};

	const u8 g[] = {
		0x5E, 0x5C, 0xBA, 0x99, 0x2E, 0x0A, 0x68, 0x0D, 0x88, 0x5E, 0xB9, 0x03, 0xAE, 0xA7, 0x8E, 0x4A, 0x45, 0xA4, 0x69, 0x10, 0x3D, 0x44, 0x8E, 0xDE, 0x3B, 0x7A, 0xCC, 0xC5,
		0x4D, 0x52, 0x1E, 0x37, 0xF8, 0x4A, 0x4B, 0xDD, 0x5B, 0x06, 0xB0, 0x97, 0x0C, 0xC2, 0xD2, 0xBB, 0xB7, 0x15, 0xF7, 0xB8, 0x28, 0x46, 0xF9, 0xA0, 0xC3, 0x93, 0x91, 0x4C,
		0x79, 0x2E, 0x6A, 0x92, 0x3E, 0x21, 0x17, 0xAB, 0x80, 0x52, 0x76, 0xA9, 0x75, 0xAA, 0xDB, 0x52, 0x61, 0xD9, 0x16, 0x73, 0xEA, 0x9A, 0xAF, 0xFE, 0xEC, 0xBF, 0xA6, 0x18,
		0x3D, 0xFC, 0xB5, 0xD3, 0xB7, 0x33, 0x2A, 0xA1, 0x92, 0x75, 0xAF, 0xA1, 0xF8, 0xEC, 0x0B, 0x60, 0xFB, 0x6F, 0x66, 0xCC, 0x23, 0xAE, 0x48, 0x70, 0x79, 0x1D, 0x59, 0x82,
		0xAA, 0xD1, 0xAA, 0x94, 0x85, 0xFD, 0x8F, 0x4A, 0x60, 0x12, 0x6F, 0xEB, 0x2C, 0xF0, 0x5D, 0xB8, 0xA7, 0xF0, 0xF0, 0x9B, 0x33, 0x97, 0xF3, 0x93, 0x7F, 0x2E, 0x90, 0xB9,
		0xE5, 0xB9, 0xC9, 0xB6, 0xEF, 0xEF, 0x64, 0x2B, 0xC4, 0x83, 0x51, 0xC4, 0x6F, 0xB1, 0x71, 0xB9, 0xBF, 0xA9, 0xEF, 0x17, 0xA9, 0x61, 0xCE, 0x96, 0xC7, 0xE7, 0xA7, 0xCC,
		0x3D, 0x3D, 0x03, 0xDF, 0xAD, 0x10, 0x78, 0xBA, 0x21, 0xDA, 0x42, 0x51, 0x98, 0xF0, 0x7D, 0x24, 0x81, 0x62, 0x2B, 0xCE, 0x45, 0x96, 0x9D, 0x9C, 0x4D, 0x60, 0x63, 0xD7,
		0x2A, 0xB7, 0xA0, 0xF0, 0x8B, 0x2F, 0x49, 0xA7, 0xCC, 0x6A, 0xF3, 0x35, 0xE0, 0x8C, 0x47, 0x20, 0xE3, 0x14, 0x76, 0xB6, 0x72, 0x99, 0xE2, 0x31, 0xF8, 0xBD, 0x90, 0xB3,
		0x9A, 0xC3, 0xAE, 0x3B, 0xE0, 0xC6, 0xB6, 0xCA, 0xCE, 0xF8, 0x28, 0x9A, 0x2E, 0x28, 0x73, 0xD5, 0x8E, 0x51, 0xE0, 0x29, 0xCA, 0xFB, 0xD5, 0x5E, 0x68, 0x41, 0x48, 0x9A,
		0xB6, 0x6B, 0x5B, 0x4B, 0x9B, 0xA6, 0xE2, 0xF7, 0x84, 0x66, 0x08, 0x96, 0xAF, 0xF3, 0x87, 0xD9, 0x28, 0x44, 0xCC, 0xB8, 0xB6, 0x94, 0x75, 0x49, 0x6D, 0xE1, 0x9D, 0xA2,
		0xE5, 0x82, 0x59, 0xB0, 0x90, 0x48, 0x9A, 0xC8, 0xE6, 0x23, 0x63, 0xCD, 0xF8, 0x2C, 0xFD, 0x8E, 0xF2, 0xA4, 0x27, 0xAB, 0xCD, 0x65, 0x75, 0x0B, 0x50, 0x6F, 0x56, 0xDD,
		0xE3, 0xB9, 0x88, 0x56, 0x7A, 0x88, 0x12, 0x6B, 0x91, 0x4D, 0x78, 0x28, 0xE2, 0xB6, 0x3A, 0x6D, 0x7E, 0xD0, 0x74, 0x7E, 0xC5, 0x9E, 0x0E, 0x0A, 0x23, 0xCE, 0x7D, 0x8A,
		0x74, 0xC1, 0xD2, 0xC2, 0xA7, 0xAF, 0xB6, 0xA2, 0x97, 0x99, 0x62, 0x0F, 0x00, 0xE1, 0x1C, 0x33, 0x78, 0x7F, 0x7D, 0xED, 0x3B, 0x30, 0xE1, 0xA2, 0x2D, 0x09, 0xF1, 0xFB,
		0xDA, 0x1A, 0xBB, 0xBF, 0xBF, 0x25, 0xCA, 0xE0, 0x5A, 0x13, 0xF8, 0x12, 0xE3, 0x45, 0x63, 0xF9, 0x94, 0x10, 0xE7, 0x3B,
	};

	const u8 x[] = {
		0x3A, 0xBC, 0x15, 0x87, 0x29, 0x7C, 0xE7, 0xB9, 0xEA, 0x1A, 0xD6, 0x65, 0x1C, 0xF2, 0xBC, 0x4D, 0x7F, 0x92, 0xED, 0x25, 0xCA, 0xBC, 0x85, 0x53, 0xF5, 0x67, 0xD1, 0xB4,
		0x0E, 0xBB, 0x87, 0x64,
	};

	const u8 y[] = {
		0x8b, 0x89, 0x1c, 0x86, 0x92, 0xd3, 0xde, 0x87, 0x58, 0x79, 0x39, 0x0f, 0x26, 0x98, 0xb2, 0x6f, 0xbe, 0xcc, 0xa6, 0xb0, 0x75, 0x53, 0x5d, 0xce, 0x6b, 0x0c, 0x86, 0x25, 0x77, 0xf9, 0xfa, 0x0d, 0xef, 0x60, 0x74, 0xe7, 0xa7, 0x62, 0x41, 0x21, 0x22, 0x4a, 0x59, 0x58, 0x96, 0xab, 0xd4, 0xcd, 0xa5, 0x6b, 0x2c, 0xef, 0xb9, 0x42, 0xe0, 0x25, 0xd2, 0xa4, 0x28, 0x2f, 0xfa, 0xa9, 0x8a, 0x48, 0xcd, 0xb4, 0x7e, 0x1a, 0x6f, 0xcb, 0x5c, 0xfb, 0x39, 0x3e, 0xf3, 0x5a, 0xf9, 0xdf, 0x91, 0x31, 0x02, 0xbb, 0x30, 0x3c, 0x2b, 0x5c, 0x36, 0xc3, 0xf8, 0xfc, 0x04, 0xed, 0x7b, 0x8b, 0x69, 0xfe, 0xfe, 0x0c, 0xf3, 0xe1, 0xfc, 0x05, 0xcf, 0xa7, 0x13, 0xb3, 0x43, 0x5b, 0x26, 0x56, 0xe9, 0x13, 0xba, 0x88, 0x74, 0xae, 0xa9, 0xf9, 0x36, 0x00, 0x6a, 0xeb, 0x44, 0x8b, 0xcd, 0x00, 0x5d, 0x18, 0xec, 0x35, 0x62, 0xa3, 0x3d, 0x04, 0xcf, 0x25, 0xc8, 0xd3, 0xd6, 0x98, 0x44, 0x34, 0x34, 0x42, 0xfa, 0x3d, 0xb7, 0xde, 0x61, 0x8c, 0x5e, 0x2d, 0xa0, 0x64, 0x57, 0x3e, 0x61, 0xe6, 0xd5, 0x58, 0x1b, 0xfb, 0x69, 0x4a, 0x23, 0xac, 0x87, 0xfd, 0x5b, 0x52, 0xd6, 0x2e, 0x95, 0x4e, 0x13, 0x76, 0xdb, 0x8d, 0xdb, 0x52, 0x4f, 0xfc, 0x0d, 0x46, 0x9d, 0xf9, 0x78, 0x79, 0x2e, 0xe4, 0x41, 0x73, 0x8e, 0x5d, 0xb0, 0x5a, 0x7d, 0xc4, 0x3e, 0x94, 0xc1, 0x1a, 0x2e, 0x7a, 0x4f, 0xbe, 0x38, 0x30, 0x71, 0xfa, 0x36, 0xd2, 0xa7, 0xec, 0x8a, 0x93, 0x88, 0xfe, 0x1c, 0x4f, 0x79, 0x88, 0x8a, 0x99, 0xd3, 0xb6, 0x10, 0x56, 0x97, 0xc2, 0x55, 0x6b, 0x79, 0xbb, 0x4d, 0x7e, 0x78, 0x1c, 0xeb, 0xb3, 0xd4, 0x86, 0x6a, 0xd8, 0x25, 0xa5, 0xe8, 0x30, 0x84, 0x60, 0x72, 0x28, 0x9f, 0xdb, 0xc9, 0x41, 0xfa, 0x67, 0x9c, 0xa8, 0x2f, 0x5f, 0x78, 0xb7, 0x46, 0x1b, 0x24, 0x04, 0xdb, 0x88, 0x3d, 0x21, 0x5f, 0x4e, 0x06, 0x76, 0xcf, 0x54, 0x93, 0x95, 0x0a, 0xc5, 0x59, 0x16, 0x97, 0xbf, 0xea, 0x8d, 0x1e, 0xe6, 0xec, 0x01, 0x6b, 0x89, 0xba, 0x51, 0xca, 0xfb, 0x5f, 0x9c, 0x84, 0xc9, 0x89, 0xfa, 0x11, 0x73, 0x75, 0xe9, 0x45, 0x78, 0xf2, 0x8b, 0xe0, 0xb3, 0x4c, 0xe0, 0x54, 0x5d, 0xa4, 0x62, 0x66, 0xfd, 0x77, 0xf6, 0x2d, 0x8f, 0x2c, 0xee, 0x92, 0xab, 0x77, 0x01, 0x2a, 0xfe, 0xbc, 0x11, 0x00, 0x89, 0x85, 0xa8, 0x21, 0xcd, 0x2d, 0x97, 0x8c, 0x7e, 0x6f, 0xe7, 0x49, 0x9d, 0x1a, 0xaf, 0x8d, 0xe6, 0x32, 0xc2, 0x1b, 0xb4, 0x8c, 0xa5, 0xcb, 0xf9, 0xf3, 0x10, 0x98, 0xfd, 0x3f, 0xd3, 0x85, 0x4c, 0x49, 0xa6, 0x5d, 0x92, 0x01, 0x74, 0x4a, 0xac, 0xe5, 0x40, 0x35, 0x49, 0x74, 0xf9,
	};

	const u8 msg[] = "abc";

	const u8 nonce[] = {
		0xA6, 0x90, 0x2C, 0x1E, 0x6E, 0x39, 0x43, 0xC5, 0x62, 0x80, 0x61, 0x58, 0x8A, 0x8B, 0x00, 0x7B, 0xCC, 0xEA, 0x91, 0xDB, 0xF1, 0x29, 0x15, 0x48, 0x3F, 0x04, 0xB2, 0x4A,
		0xB0, 0x67, 0x8B, 0xEE,
	};

	dsa_priv_key priv;
	dsa_pub_key pub;
	dsa_pub_key pub2;
	u8 sig[32*2] = { 0 };

	FORCE_USED_VAR(argc);
	FORCE_USED_VAR(argv);

	/* Sanity check on size for DSA.
	 * NOTE: the double parentheses are here to handle -Wunreachable-code
	 */
	if((NN_USABLE_MAX_BIT_LEN) < (4096)){
		ext_printf("Error: you seem to have compiled libecc with usable NN size < 4096, not suitable for DSA.\n");
		ext_printf("  => Please recompile libecc with EXTRA_CFLAGS=\"-DUSER_NN_BIT_LEN=4096\"\n");
		ext_printf("     This will increase usable NN for proper DSA up to 4096 bits.\n");
		ext_printf("     Then recompile the current examples with the same EXTRA_CFLAGS=\"-DUSER_NN_BIT_LEN=4096\" flag and execute again!\n");
		/* NOTE: ret = 0 here to pass self tests even if the library is not compatible */
		ret = 0;
		goto err;
	}


	ret = dsa_import_priv_key(&priv, p, sizeof(p), q, sizeof(q), g, sizeof(g), x, sizeof(x)); EG(ret, err);
	ret = dsa_import_pub_key(&pub, p, sizeof(p), q, sizeof(q), g, sizeof(g), y, sizeof(y)); EG(ret, err);
	ret = dsa_compute_pub_from_priv(&pub2, &priv); EG(ret, err);

	nn_print("y", &(pub2.y));

	ret = dsa_sign(&priv, msg, sizeof(msg)-1, nonce, sizeof(nonce), sig, sizeof(sig), HASH_SHA256); EG(ret, err);

	buf_print("sig", sig, sizeof(sig));

	ret = dsa_verify(&pub, msg, sizeof(msg)-1, sig, sizeof(sig), HASH_SHA256);
	ext_printf("Signature result %d\n", ret);

err:
	return ret;
}
#endif
